CREATE OR REPLACE PACKAGE sim2_export_result_pkg IS
/* Updated 2/13/02        by SF
   Activity, Trip-QC, and Trip Time Zone fixes
   
*/   
  -- Author  : KHENAS
  -- Created : 11/16/00 4:24:36 PM
  -- Purpose : load results data from SIM into STORET

  -- Public variable declarations
  v_export_log_seq          NUMBER(10);
  v_cur_fieldset_is_nbr     NUMBER(8);
  v_cur_stavst_is_nbr       NUMBER(8);
  v_cur_activity_is_nbr     NUMBER(8);
  v_cur_trip_is_nbr         NUMBER(8);
  v_cur_trip_qc_is_nbr      NUMBER(8);
  v_cur_result_is_nbr       NUMBER(8);
  v_cur_tsmgntxt_is_nbr     NUMBER(8);
  
  v_stavst_no               NUMBER(3);  
  v_last_station            VARCHAR2(15);  
  v_activity_cnt            NUMBER(10) := 0;
  v_result_cnt              NUMBER(10) := 0;
  v_single_user             NUMBER(1); -- Single user environment?
    
  -- Public function and procedure declarations
  FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2;

  FUNCTION GET_UOM_IS_NBR (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2, p_chdef_is_number IN NUMBER)
     RETURN NUMBER;     

  FUNCTION validate_sample_frac (p_sample_frac_nm IN VARCHAR2, p_tsrchdef_is_nbr IN NUMBER,
                                 p_tsrchar_is_nbr IN NUMBER) RETURN VARCHAR2;
                               
  FUNCTION get_next_visit_is_number (p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_next_activity_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_next_result_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER;

  FUNCTION get_value_msr (p_result_text IN VARCHAR2) RETURN NUMBER;

  PROCEDURE add_station_to_trip(p_trip_is_nbr IN NUMBER,p_sta_is_nbr IN NUMBER,p_org_id IN VARCHAR2);

  PROCEDURE add_project_to_trip(p_trip_is_nbr IN NUMBER, p_proj_is_nbr IN NUMBER, p_org_id IN VARCHAR2);

  PROCEDURE SIM_EXPORT (p_org_id  IN VARCHAR2, p_import_seq IN NUMBER);
  
  PROCEDURE EXPORT_RESULTS  (p_fa_seq IN NUMBER, p_fat_desc IN VARCHAR2, p_activity_is_nbr IN NUMBER,
                             p_activity_org_id IN VARCHAR2, p_tripqc_is_nbr IN NUMBER, p_tripqc_org_id IN VARCHAR2,
                             p_lab_is_number IN NUMBER,p_lab_org_id IN VARCHAR2, p_org_id IN VARCHAR2, 
                             p_export_log_seq IN NUMBER, p_act_result_cnt IN OUT NUMBER, p_time_zone IN VARCHAR2);

  PROCEDURE runtime;
  

END sim2_export_result_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim2_export_result_pkg IS

  e_a              EXCEPTION;

  -- Function and procedure implementations

/******************************************************************************************************/
FUNCTION get_allowable_value(p_alval_seq IN NUMBER) RETURN VARCHAR2 IS

  v_item_name         VARCHAR2(80);

  CURSOR c_get_alval_name(p_alval_seq IN NUMBER) IS
    SELECT SIA_ITEM FROM SIM_IMP_ALLOW_VALUES WHERE SIA_SEQ = p_alval_seq;

BEGIN

  OPEN c_get_alval_name(p_alval_seq);
  FETCH c_get_alval_name INTO v_item_name;
    IF c_get_alval_name%NOTFOUND THEN
      v_item_name := NULL;
      sim2_export_pkg.write_error(v_export_log_seq, 'Could not find allowable value with sequence: '||p_alval_seq);
    END IF;
  CLOSE c_get_alval_name;

  RETURN v_item_name;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting allowable value sequence.');

END get_allowable_value;

/******************************************************************************************************/
FUNCTION GET_UOM_IS_NBR (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2, p_chdef_is_number IN NUMBER)
   RETURN NUMBER IS

   TYPE t_uom_record IS RECORD (
      uom_is_nbr        NUMBER(8),
      uom_name             VARCHAR2(10));
   
   v_uom_info              t_uom_record;
   
   e_unk_uom               EXCEPTION;
   e_uom_sp                EXCEPTION;

   CURSOR c_get_is_nbr (p_org_id IN VARCHAR2, p_uom_name IN VARCHAR2) IS
      SELECT TSRUOM_IS_NUMBER, SHORT_FORM_NAME FROM TSRUOM WHERE TSRUOM_ORG_ID = RPAD(p_org_id,8) AND
      UPPER(SHORT_FORM_NAME) = UPPER(RPAD(p_uom_name,10));

BEGIN

   OPEN c_get_is_nbr(p_org_id, p_uom_name);
   FETCH c_get_is_nbr INTO v_uom_info;
   IF c_get_is_nbr%NOTFOUND THEN
      v_uom_info.uom_is_nbr := NULL;
      RAISE e_unk_uom;
   ELSE
      IF v_uom_info.uom_name != p_uom_name THEN
         RAISE e_uom_sp;
      END IF;
   END IF;
   CLOSE c_get_is_nbr;

   RETURN v_uom_info.uom_is_nbr;
   
EXCEPTION
   WHEN e_unk_uom THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Unknown Unit of Measure: '||p_uom_name);
     RETURN v_uom_info.uom_is_nbr;

   WHEN e_uom_sp THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Fix uom spelling in TSRCHDEF - is_number:'||p_chdef_is_number||', unit of measure misspelled: '||p_uom_name);   
     RETURN v_uom_info.uom_is_nbr;

END GET_UOM_IS_NBR;

/******************************************************************************************************/
FUNCTION validate_sample_frac (p_sample_frac_nm IN VARCHAR2, p_tsrchdef_is_nbr IN NUMBER,
                               p_tsrchar_is_nbr IN NUMBER) RETURN VARCHAR2 IS

  v_sample_frac               VARCHAR2(40);
  v_samp_req_cd               VARCHAR2(1);
  
  e_sample_frac_sp            EXCEPTION;
  
  CURSOR c_get_samp_req_cd (p_tsrchar_is_nbr IN NUMBER) IS
    SELECT SAMP_FRAC_REQ_CD FROM TSRCHAR WHERE tsrchar_is_number = p_tsrchar_is_nbr;

  CURSOR c_check_def_frac IS
    SELECT FIELD_VALUE FROM TSMPRMVL 
    WHERE FIELD_NAME = RPAD('SMPL_FRAC_TYPE_NM',20) AND TABLE_NAME = 'TSRCHDEF';

BEGIN

  /* set sample fraction to null if it is full of spaces */
  IF RPAD(p_sample_frac_nm,15) = RPAD(' ',15) OR p_sample_frac_nm IS NULL THEN
    v_sample_frac := NULL;
  ELSE
    v_sample_frac := p_sample_frac_nm;
  END IF;

  OPEN c_get_samp_req_cd(p_tsrchar_is_nbr);
  FETCH c_get_samp_req_cd INTO v_samp_req_cd;
    IF v_samp_req_cd = 'Y' OR v_sample_frac IS NOT NULL THEN

      FOR v_fraction IN c_check_def_frac LOOP
        IF v_fraction.FIELD_VALUE = RPAD(p_sample_frac_nm,40) THEN
          RETURN RTRIM(v_fraction.FIELD_VALUE);
        ELSIF UPPER(v_fraction.FIELD_VALUE) = UPPER(RPAD(p_sample_frac_nm,40)) THEN
          v_sample_frac := RTRIM(v_fraction.FIELD_VALUE);
          CLOSE c_get_samp_req_cd;
          RAISE e_sample_frac_sp;    
        END IF;
      END LOOP;

      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Invalid fraction in TSRCHDEF_IS_NUMBER:'||p_tsrchdef_is_nbr||', sample fraction, '||p_sample_frac_nm||', not loaded into TSRRSULT');     
      RETURN NULL;
  
    END IF;
  CLOSE c_get_samp_req_cd;
  
  RETURN NULL;
  
EXCEPTION
  WHEN e_sample_frac_sp THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Fix fraction spelling in TSRCHDEF - is_number:'||p_tsrchdef_is_nbr||', sample fraction: '||p_sample_frac_nm);   
     RETURN v_sample_frac;    
     
  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in validate sample fraction.');   
     RETURN v_sample_frac;         

END validate_sample_frac;

/******************************************************************************************************/
FUNCTION GET_FIELD_SET_IS_NBR (p_org_id VARCHAR2, p_visit_is_number NUMBER, 
                               p_id_code IN VARCHAR2, p_fdset_name IN VARCHAR2) 
                               RETURN NUMBER IS

   /* this function looks to see if there is already a field set with the same id-code set up for a visit.
      if it is, it returns the is-number of the field set.  If not, it creates a field set record and returns it's 
      IS_number.    SNG 4/21/2000    
   */

   v_field_set_is_nbr NUMBER(8);
   v_table_name       VARCHAR2(8);

   CURSOR c_field_set_is_nbr (p_org_id VARCHAR2, p_visit_is_number NUMBER, p_id_code VARCHAR2) IS
      SELECT tsrfdset_is_number FROM TSRFDSET WHERE 
         ID_CODE = RPAD(p_id_code,10) AND 
         TSRFDSET_ORG_ID = RPAD(p_org_id,8) AND
         TSRSTVST_IS_NUMBER = p_visit_is_number;


BEGIN

   OPEN c_field_set_is_nbr (p_org_id , p_visit_is_number , p_id_code );
   FETCH c_field_set_is_nbr INTO v_field_set_is_nbr;
   IF c_field_set_is_nbr%NOTFOUND THEN
      
       /* not found, so set one up */

       v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

       /* if the next seq number is available, use it and increment, otherwise query it from the database */
       IF v_cur_fieldset_is_nbr IS NULL OR v_single_user = 0 THEN
           v_table_name := 'TSRFDSET';
           v_field_set_is_nbr :=  sim2_export_pkg.get_eisn_number(v_table_name,p_org_id);
           v_cur_fieldset_is_nbr := v_field_set_is_nbr;
       ELSE
           v_field_set_is_nbr := v_cur_fieldset_is_nbr + 1;
           v_cur_fieldset_is_nbr := v_field_set_is_nbr;

       END IF;
       

       INSERT INTO TSRFDSET (
         TSRFDSET_ORG_ID, 
         TSRFDSET_IS_NUMBER, 
         ID_CODE, 
         NAME, 
         D_USERID_CODE,
         D_LAST_UPDATE_TS, 
         TSRSTVST_IS_NUMBER, 
         TSRSTVST_ORG_ID) 
       VALUES (
         RPAD(p_org_id,8), 
         v_field_set_is_nbr, 
         p_id_code, 
         p_fdset_name,
         'STORUSER', 
         SYSDATE, 
         p_visit_is_number, 
         RPAD(p_org_id,8));

       IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
         sim2_export_pkg.UPDATE_TSMEISN(rpad(p_org_id,8),'TSRFDSET', v_field_set_is_nbr);
         COMMIT;
       END IF;
        
       COMMIT;

   END IF;

   CLOSE c_field_set_is_nbr;
    

   RETURN v_field_set_is_nbr;
   
EXCEPTION
  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get_field_set_is_nbr.');   
     RETURN NULL;       

END GET_FIELD_SET_IS_NBR;

/******************************************************************************************************/
FUNCTION get_sample_pres_info(p_tsrsdp_is_nbr IN NUMBER, p_tsrsdp_org_id IN VARCHAR2,
  p_activity_id IN VARCHAR2) RETURN TSRSMPLE%ROWTYPE IS

  v_sample_rec           TSRSMPLE%ROWTYPE;
  v_sample_copy_rec      TSRSMPLE%ROWTYPE;
  
  e_bad_sample_pres      EXCEPTION;
  CURSOR c_get_sample_pres_info IS
    SELECT CONTAINER_TYPE_NM, CONTAINER_COLOR, CONTAINER_SIZE_MSR, CONTAINER_SIZE_UN,
      TEMP_PRESRV_TYPE, PRESRV_STRGE_PRCDR FROM TSRSDP 
      WHERE TSRSDP_IS_NUMBER = p_tsrsdp_is_nbr AND TSRSDP_ORG_ID = RPAD(p_tsrsdp_org_id,8);

BEGIN
  /* kms (5/22/01) - this function gets all of the sample preservation, storage, and transport 
    information on the activity and returns it in the form of a sample rowtype.  This is
    performed in this manner since STORET does not store this information relationally. */
  
  OPEN c_get_sample_pres_info;
  FETCH c_get_sample_pres_info INTO v_sample_rec.container_type_nm,v_sample_rec.container_color,
    v_sample_rec.container_size_msr,v_sample_rec.container_size_un,v_sample_rec.temp_preservn_type,
    v_sample_rec.presrv_strge_prcdr;
    IF c_get_sample_pres_info%NOTFOUND THEN
      CLOSE c_get_sample_pres_info;
      RAISE e_bad_sample_pres;
    END IF;
  CLOSE c_get_sample_pres_info;

  RETURN v_sample_rec;
  
EXCEPTION
  WHEN e_bad_sample_pres THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Activity '||p_activity_id||': Unable to find sample preservation information - invalid foreign key.');   
     RETURN v_sample_copy_rec; 
  
  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next tsmgntxt is nbr.');   
     RETURN v_sample_copy_rec; 

END get_sample_pres_info;

/******************************************************************************************************/
FUNCTION get_instructions(p_import_seq IN NUMBER, p_instruction_type IN VARCHAR2) RETURN VARCHAR2 IS
          
  v_imp_cfg_seq          NUMBER(10);
  v_instr_choice         VARCHAR2(15);    
  
  e_no_cfg               EXCEPTION;
  
  CURSOR c_get_cfg_seq(p_import_seq IN NUMBER) IS
    SELECT SIL_SICC_SEQ FROM SIM_IMPORT_LOG WHERE SIL_SEQ = p_import_seq;
          
  CURSOR c_get_cfg_instr(p_cfg_seq IN NUMBER, p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_CFG_INSTRS, SIM_IMP_INSTR_CHOICES, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ      
      AND SII_CODE = p_instr_type AND SICI_SICC_SEQ = p_cfg_seq;

  CURSOR c_get_instr_default(p_instr_type IN VARCHAR2) IS
    SELECT SIIC_CODE FROM SIM_IMP_INSTR_CHOICES, SIM_IMP_CFG_INSTRS, SIM_IMP_INSTRS
      WHERE SICI_SIIC_SEQ = SIIC_SEQ AND SIIC_SII_SEQ = SII_SEQ      
      AND SII_CODE = p_instr_type AND SIIC_DFLT_FLAG = 1;

      
BEGIN

  /* get the configuration for this import */
  OPEN c_get_cfg_seq(p_import_seq);
  FETCH c_get_cfg_seq INTO v_imp_cfg_seq;
    IF c_get_cfg_seq%NOTFOUND THEN
      v_imp_cfg_seq := NULL;
      RAISE e_no_cfg;
    END IF;
  CLOSE c_get_cfg_seq;

  /* kms - get the instruction choice for this item */
  OPEN c_get_cfg_instr(v_imp_cfg_seq, p_instruction_type);
  FETCH c_get_cfg_instr INTO v_instr_choice;
    IF c_get_cfg_instr%NOTFOUND THEN

      /* no choice could be found, get default choice */
      OPEN c_get_instr_default(p_instruction_type);
      FETCH c_get_instr_default INTO v_instr_choice;
      CLOSE c_get_instr_default;
    
    END IF;
  CLOSE c_get_cfg_instr;

  RETURN v_instr_choice;
  
EXCEPTION
  WHEN e_no_cfg THEN
    RAISE_APPLICATION_ERROR(-20101, 'Configuration could not be found for this import.');
    RETURN NULL;

  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get_instructions.');   
     RETURN NULL;
     
END get_instructions;

/******************************************************************************************************/
FUNCTION get_next_visit_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS
   
   v_next_visit_is_nbr NUMBER(8);
   
BEGIN

 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

   IF  v_cur_stavst_is_nbr IS NULL OR v_single_user = 0 THEN
   /* get the next avail is nbr to use */
      v_next_visit_is_nbr := sim2_export_pkg.get_eisn_number('TSRSTVST',p_org_id);
      v_cur_stavst_is_nbr := v_next_visit_is_nbr ;
   ELSE
     v_next_visit_is_nbr := v_cur_stavst_is_nbr  + 1;
     v_cur_stavst_is_nbr := v_next_visit_is_nbr;
   END IF;
   
   RETURN v_next_visit_is_nbr;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next visit is number.');   
     RETURN NULL;
   
END get_next_visit_is_number;

/******************************************************************************************************/
FUNCTION get_next_activity_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

BEGIN
 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

   IF v_cur_activity_is_nbr IS NULL OR v_single_user = 0 THEN
      v_cur_activity_is_nbr := sim2_export_pkg.get_eisn_number('TSRFDACT',p_org_id);
   ELSE
      v_cur_activity_is_nbr := v_cur_activity_is_nbr + 1;
   END IF;

   RETURN v_cur_activity_is_nbr ;
   
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next activity is nbr.');   
     RETURN NULL;   

END get_next_activity_is_nbr;

/******************************************************************************************************/
FUNCTION get_next_trip_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

BEGIN
 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

   IF v_cur_trip_is_nbr IS NULL OR v_single_user = 0 THEN
      v_cur_trip_is_nbr := sim2_export_pkg.get_eisn_number('TSRTRIP ',p_org_id);
   ELSE
      v_cur_trip_is_nbr := v_cur_trip_is_nbr + 1;
   END IF;

   RETURN v_cur_trip_is_nbr ;
   
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next trip is nbr.');   
     RETURN NULL;   

END get_next_trip_is_nbr;

/******************************************************************************************************/
FUNCTION get_next_trip_qc_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

BEGIN
 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

   IF v_cur_trip_qc_is_nbr IS NULL OR v_single_user = 0 THEN
      v_cur_trip_qc_is_nbr := sim2_export_pkg.get_eisn_number('TSRFQS  ',p_org_id);
   ELSE
      v_cur_trip_qc_is_nbr := v_cur_trip_qc_is_nbr + 1;
   END IF;

   RETURN v_cur_trip_qc_is_nbr ;
   
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next trip qc is nbr.');   
     RETURN NULL;   

END get_next_trip_qc_is_nbr;

/******************************************************************************************************/
FUNCTION get_next_result_is_nbr (p_org_id IN VARCHAR2) RETURN NUMBER IS

BEGIN
 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

   IF v_cur_result_is_nbr IS NULL OR v_single_user = 0 THEN
     v_cur_result_is_nbr := sim2_export_pkg.get_eisn_number('TSRRSULT',p_org_id);
   ELSE
      v_cur_result_is_nbr := v_cur_result_is_nbr + 1;
   END IF;
  
   RETURN v_cur_result_is_nbr ;
   
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next result is nbr.');   
     RETURN NULL;   

END get_next_result_is_nbr;

/******************************************************************************************************/
FUNCTION get_next_tsmgntxt_nbr(p_org_id IN VARCHAR2) RETURN NUMBER IS

BEGIN
 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

  IF v_cur_tsmgntxt_is_nbr IS NULL OR v_single_user = 0 THEN
    v_cur_tsmgntxt_is_nbr := sim2_export_pkg.get_eisn_number('TSMGNTXT',p_org_id);
  ELSE
    v_cur_tsmgntxt_is_nbr := v_cur_tsmgntxt_is_nbr + 1;
  END IF;

  RETURN v_cur_tsmgntxt_is_nbr;
  
EXCEPTION
  WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next tsmgntxt is nbr.');   
     RETURN NULL; 

END get_next_tsmgntxt_nbr;

/******************************************************************************************************/
FUNCTION get_value_msr (p_result_text IN VARCHAR2) RETURN NUMBER IS

  v_result_value       TSRRSULT.VALUE_MEASURE%TYPE;

BEGIN

   BEGIN
      /* see if the result is a number - translate result will be null if it is */
      IF TRANSLATE(p_result_text, 'a1234567890.- ', 'a') IS NULL THEN
         /* see if the first part of the number is longer that 8 */
         IF INSTR(p_result_text,'.') > 9 THEN
            /* set numeric portion of the result to an unusual number, keep the text value the same */
            v_result_value := 99999999.99999;
            sim2_export_pkg.write_error(v_export_log_seq, 'Number too large for STORET value column, '||
                                       'set to ''99999999.99999'': '||p_result_text);
         /* convert char to number and round decimals to five places if the number will fit */
         ELSIF INSTR(p_result_text,'.') != 0 THEN
            v_result_value := ROUND(TO_NUMBER(p_result_text),5);                            
         ELSE 
            /* if the number does not have a decimal and is larger than 8, set to unusual number */
            IF LENGTH(p_result_text) > 8 THEN
               v_result_value := 99999999.99999;
               sim2_export_pkg.write_error(v_export_log_seq, 'Number too large for STORET value column, '||
                                       'set to ''99999999.99999'': '||p_result_text);            
            /* else just convert it to a number */
            ELSE
               v_result_value := TO_NUMBER(p_result_text);
            END IF;
         END IF;            
      ELSE 
         /* if the data in the result text field is not a number, set result_value to NULL */
         v_result_value := NULL;
      END IF;
      
--      sim2_export_pkg.write_error(v_export_log_seq, 'Value text converted to: '||v_result_value);
      
   EXCEPTION
      WHEN OTHERS THEN
         v_result_value := NULL;
         sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - converting result to number: '||v_result_value);
   END;
   
   RETURN v_result_value;
   
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get value measure.');   
     RETURN NULL;             

END get_value_msr;

/******************************************************************************************************/
PROCEDURE get_trip_info(p_trip_seq IN NUMBER, p_trip_rec OUT TSRTRIP%ROWTYPE) IS

  CURSOR c_get_trip_info (p_trip_seq IN NUMBER) IS
    SELECT STP_TRIP_ID, STP_TRIP_ID, STP_START_DATE, STP_START_TIME, STP_END_DATE, STP_END_TIME, STP_TSMORGAN_IS_NUMBER
      FROM SIM_TRIPS WHERE STP_SEQ = p_trip_seq;

BEGIN
 
  /* kms (6/4/01) - get info about this trip from the SIM_TRIPS table */
  OPEN c_get_trip_info(p_trip_seq);
  FETCH c_get_trip_info INTO p_trip_rec.ID_CODE, p_trip_rec.NAME, p_trip_rec.start_date, 
                             p_trip_rec.start_time, p_trip_rec.end_date, p_trip_rec.end_time, 
                             p_trip_rec.TSMORGAN_IS_NUMBER;
  CLOSE c_get_trip_info;  

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get trip info.');   
      
END get_trip_info;

/******************************************************************************************************/
PROCEDURE get_visit_info(p_visit_seq IN NUMBER, p_visit_rec OUT TSRSTVST%ROWTYPE, p_visit_id OUT VARCHAR2) IS

  CURSOR c_get_visit_info (p_visit_seq IN NUMBER) IS
    SELECT SSV_ID, SSV_START_DATE, SSV_START_TIME, SSV_END_DATE, SSV_END_TIME
      FROM SIM_STATION_VISITS WHERE SSV_SEQ = p_visit_seq;

BEGIN
 
  /* kms (6/4/01) - get info about this trip from the SIM_TRIPS table */
  OPEN c_get_visit_info(p_visit_seq);
  FETCH c_get_visit_info INTO p_visit_id, p_visit_rec.arrival_date, 
                              p_visit_rec.arrival_time, p_visit_rec.departure_date, 
                              p_visit_rec.departure_time;
  CLOSE c_get_visit_info;  

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get visit info.');   
      
END get_visit_info;

/******************************************************************************************************/
FUNCTION get_next_visit_id(p_trip_is_nbr IN NUMBER, p_trip_org_id IN VARCHAR2, 
                           p_statn_is_nbr IN NUMBER, p_statn_org_id IN VARCHAR2) RETURN NUMBER IS

  v_max_id          NUMBER(3) := 0;
  
  CURSOR c_get_max_visit_id (p_trip_is_nbr IN NUMBER, p_trip_org_id IN VARCHAR2, 
                             p_statn_is_nbr IN NUMBER, p_statn_org_id IN VARCHAR2) IS
    SELECT ID_NUMBER FROM TSRSTVST WHERE TSRTRIP_IS_NUMBER = p_trip_is_nbr
      AND TSRTRIP_ORG_ID = RPAD(p_trip_org_id,8) AND TSMSTATN_IS_NUMBER = p_statn_is_nbr
      AND TSMSTATN_ORG_ID = RPAD(p_statn_org_id,8);

BEGIN 
  /* kms (6/8/01) - get the max station visit id for this trip and station from STORET
     and increment it by 1. */
     
  FOR v_visit_id IN c_get_max_visit_id(p_trip_is_nbr,p_trip_org_id,p_statn_is_nbr,p_statn_org_id) LOOP
  
    /* kms - see if visit id is a number */
    IF TRANSLATE(v_visit_id.ID_NUMBER, 'a1234567890.- ', 'a') IS NULL THEN
    
      /* kms - if visit id is a number, see if it is larger than the known max id, if so
         set max id. */
      IF v_visit_id.ID_NUMBER > v_max_id THEN
        v_max_id := v_visit_id.ID_NUMBER;
      END IF;
    
    END IF;  
  
  END LOOP;
  
  v_max_id := v_max_id + 1;
  
  RETURN v_max_id;
  
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next visit id.'); 
     RETURN NULL;  
      
END get_next_visit_id;

/******************************************************************************************************/
PROCEDURE add_station_to_trip(p_trip_is_nbr IN NUMBER,p_sta_is_nbr IN NUMBER,p_org_id IN VARCHAR2) IS

  v_tsrtsa                  NUMBER(8);

  CURSOR c_get_tsrtsa (p_cursor_trip IN NUMBER, p_cursor_sta IN NUMBER, p_cursor_org_id IN VARCHAR2) IS
    SELECT TSRTRIP_IS_NUMBER FROM tsrtsa WHERE TSMSTATN_IS_NUMBER = p_cursor_sta
    AND TSRTRIP_IS_NUMBER = p_cursor_trip 
    AND TSRTRIP_ORG_ID = p_cursor_org_id;

BEGIN
  /* see if station is already on the trip */
  OPEN c_get_tsrtsa (p_trip_is_nbr,p_sta_is_nbr,RPAD(p_org_id,8));
  FETCH c_get_tsrtsa INTO v_tsrtsa;
     IF c_get_tsrtsa%NOTFOUND THEN        
       /* Update TSRTSA to associate station with this trip if record does not already exist.  */           
        INSERT INTO TSRTSA (       
           TSRTRIP_IS_NUMBER,
           TSRTRIP_ORG_ID,
           TSMSTATN_IS_NUMBER,
           TSMSTATN_ORG_ID,
           D_USERID_CODE,
           D_LAST_UPDATE_TS
           )
        VALUES (
           p_trip_is_nbr,                     /* TSRTRIP_IS_NUMBER */
           RPAD(p_org_id,8),                  /* TSRTRIP_ORG_ID */
           p_sta_is_nbr,                      /* TSMSTATN_IS_NUMBER */
           RPAD(p_org_id,8),                  /* TSMSTATN_ORG_ID */
           USER,                              /* D_USERID_CODE */
           SYSDATE                            /* D_LAST_UPDATE_TS */
           );  
     
        COMMIT;
     
     END IF;
  CLOSE c_get_tsrtsa;  

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add station to trip.');   
      
END add_station_to_trip;

/******************************************************************************************************/
PROCEDURE add_project_to_activity(p_act_is_nbr IN NUMBER, p_proj_is_nbr IN NUMBER,
                                  p_org_id IN VARCHAR2) IS

  v_proj_is                  NUMBER(8);
  
                                 
  CURSOR c_check_for_project(p_cursor_act IN NUMBER,p_cursor_proj IN NUMBER,p_org_id IN VARCHAR2) IS
    SELECT TSMPROJ_IS_NUMBER FROM TSRFAPRA WHERE TSMPROJ_IS_NUMBER = p_cursor_proj 
    AND TSRFDACT_IS_NUMBER = p_cursor_act AND TSRFDACT_ORG_ID = RPAD(p_org_id,8);
                             
BEGIN  
  /* see if the project is on the activity */
  OPEN c_check_for_project(p_act_is_nbr,p_proj_is_nbr,p_org_id);
  FETCH c_check_for_project INTO v_proj_is;
    IF c_check_for_project%NOTFOUND THEN
      /* if project is not on the activity then add */
      INSERT INTO TSRFAPRA(
        TSRFDACT_IS_NUMBER,
        TSRFDACT_ORG_ID,         
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,         
        D_USERID_CODE,          
        D_LAST_UPDATE_TS       
        )
      VALUES(
        p_act_is_nbr,            /* TSRFDACT_IS_NUMBER */
        p_org_id,                /* TSRFDACT_ORG_ID */        
        p_proj_is_nbr,           /* TSMPROJ_IS_NUMBER */
        p_org_id,                /* TSMPROJ_ORG_ID */        
        USER,                    /* D_USERID_CODE */         
        SYSDATE                  /* D_LAST_UPDATE_TS */      
        );
      COMMIT;  
    END IF;
  CLOSE c_check_for_project;  
  
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add project to activity.');   

END add_project_to_activity;

/******************************************************************************************************/
PROCEDURE add_project_to_trip(p_trip_is_nbr IN NUMBER, p_proj_is_nbr IN NUMBER,
                              p_org_id IN VARCHAR2) IS

  v_proj_is                  NUMBER(8);
  v_number_projects          NUMBER(8);
  
                                 
  CURSOR c_check_for_project(p_cursor_trip IN NUMBER,p_cursor_proj IN NUMBER,p_org_id IN VARCHAR2) IS
    SELECT TSMPROJ_IS_NUMBER FROM TSRTPA WHERE TSMPROJ_IS_NUMBER = p_cursor_proj 
    AND TSRTRIP_IS_NUMBER = p_cursor_trip AND TSRTRIP_ORG_ID = RPAD(p_org_id,8);
    
  CURSOR c_check_for_single_proj(p_cursor_trip IN NUMBER,p_cursor_org_id IN VARCHAR2) IS
    SELECT COUNT(TSMPROJ_IS_NUMBER) FROM TSRTPA WHERE TSRTRIP_IS_NUMBER = p_cursor_trip
    AND TSRTRIP_ORG_ID = p_cursor_org_id;
                               
                             
BEGIN  
  /* see if the project is on the trip */
  OPEN c_check_for_project(p_trip_is_nbr,p_proj_is_nbr,p_org_id);
  FETCH c_check_for_project INTO v_proj_is;
    IF c_check_for_project%NOTFOUND THEN
      /* if project is not on the trip then add */
      INSERT INTO TSRTPA(
        TSRTRIP_IS_NUMBER,
        TSRTRIP_ORG_ID,         
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,         
        D_USERID_CODE,          
        D_LAST_UPDATE_TS       
        )
      VALUES(
        p_trip_is_nbr,           /* TSRTRIP_IS_NUMBER */
        p_org_id,                /* TSRTRIP_ORG_ID */        
        p_proj_is_nbr,           /* TSMPROJ_IS_NUMBER */
        p_org_id,                /* TSMPROJ_ORG_ID */        
        USER,                    /* D_USERID_CODE */         
        SYSDATE                  /* D_LAST_UPDATE_TS */      
        );
      COMMIT;  
    END IF;
  CLOSE c_check_for_project;  
  
  /* check for single project on trip */
  OPEN c_check_for_single_proj(p_trip_is_nbr,RPAD(p_org_id,8));
  FETCH c_check_for_single_proj INTO v_number_projects;
    /* set single project ind to 'N' if more than one project is on the trip */
    IF v_number_projects > 1 THEN
      UPDATE TSRTRIP SET SINGLE_PROJ_IND = 'N' 
        WHERE TSRTRIP_IS_NUMBER = p_trip_is_nbr AND TSRTRIP_ORG_ID = RPAD(p_org_id,8);
    END IF;    
  CLOSE c_check_for_single_proj;  
  
EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add project to trip.');   

END add_project_to_trip;

/******************************************************************************************************/
FUNCTION add_station_to_project(p_proj_is_nbr IN VARCHAR2,p_sta_is_nbr IN VARCHAR2,
                                p_org_id IN VARCHAR2) RETURN NUMBER IS
                                 
  v_sta_is                   NUMBER(8);
                                 
  CURSOR c_check_for_station(p_cursor_proj IN VARCHAR2, p_cursor_sta IN VARCHAR2, p_org_id IN VARCHAR2) IS
    SELECT TSMSTATN_IS_NUMBER FROM TSMPSA WHERE TSMSTATN_IS_NUMBER = p_cursor_sta 
      AND TSMSTATN_ORG_ID = RPAD(p_org_id,8)
      AND TSMPROJ_IS_NUMBER = p_cursor_proj AND TSMPROJ_ORG_ID = RPAD(p_org_id,8);
                                 
BEGIN

  /* see if station is on project */
  OPEN c_check_for_station(p_proj_is_nbr,p_sta_is_nbr,p_org_id);
  FETCH c_check_for_station INTO v_sta_is;
  IF c_check_for_station%NOTFOUND THEN

    /* add station to project if not found */
    INSERT INTO TSMPSA (
      TSMSTATN_IS_NUMBER,
      TSMSTATN_ORG_ID,        
      TSMPROJ_IS_NUMBER,      
      TSMPROJ_ORG_ID,         
      D_USERID_CODE,          
      D_LAST_UPDT_TS         
      )
    VALUES (
      p_sta_is_nbr,          /* TSMSTATN_IS_NUMBER */
      p_org_id,              /* TSMSTATN_ORG_ID */
      p_proj_is_nbr,         /* TSMPROJ_IS_NUMBER */
      p_org_id,              /* TSMPROJ_ORG_ID */        
      USER,                  /* D_USERID_CODE */
      SYSDATE                /* D_LAST_UPDT_TS */
      );
    END IF;
  CLOSE c_check_for_station;
  
  RETURN 1;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in add station to project.');    
    RETURN 0;

END add_station_to_project;


/******************************************************************************************************/
FUNCTION add_person_to_project(p_person_is_nbr IN NUMBER,p_person_org_id IN VARCHAR2, 
                               p_project_is_nbr IN NUMBER,p_org_id IN VARCHAR2) RETURN NUMBER IS
                               
  v_dummy                      NUMBER(8);                               
                               
  CURSOR c_check_for_person_role(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2,
                                 p_project_is_nbr IN NUMBER, p_project_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSMPPRA WHERE TSMPERSN_IS_NUMBER = p_person_is_nbr
      AND TSMPERSN_ORG_ID = RPAD(p_person_org_id, 8) AND TSMPROJ_IS_NUMBER = p_project_is_nbr
      AND TSMPROJ_ORG_ID = RPAD(p_project_org_id, 8);
      
  CURSOR c_check_for_person(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2,
                            p_project_is_nbr IN NUMBER, p_project_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSMPPA WHERE TSMPERSN_IS_NUMBER = p_person_is_nbr
      AND TSMPERSN_ORG_ID = RPAD(p_person_org_id, 8) AND TSMPROJ_IS_NUMBER = p_project_is_nbr
      AND TSMPROJ_ORG_ID = RPAD(p_project_org_id, 8);                            

  CURSOR c_check_for_role(p_person_is_nbr IN NUMBER, p_person_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSMPRA WHERE TSMPERSN_IS_NUMBER = p_person_is_nbr
      AND TSMPERSN_ORG_ID = RPAD(p_person_org_id, 8) AND TSMPROLE_IS_NUMBER = 10
      AND TSMPROLE_ORG_ID = 'ZZZZZZZZ';
      
BEGIN

  /* make sure project, person, role assignment has not already been made */
  OPEN c_check_for_person_role(p_person_is_nbr, p_person_org_id, p_project_is_nbr, p_org_id);
  FETCH c_check_for_person_role INTO v_dummy;
    IF c_check_for_person_role%NOTFOUND THEN
      
      /* if nothing found then create association using the staff/employee role */
      INSERT INTO TSMPPRA (
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        TSMPROLE_IS_NUMBER,
        TSMPROLE_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS        
        )
      VALUES (
        p_project_is_nbr,                          /* TSMPROJ_IS_NUMBER */
        p_org_id,                                  /* TSMPROJ_ORG_ID */
        p_person_is_nbr,                           /* TSMPERSN_IS_NUMBER */
        p_person_org_id,                           /* TSMPERSN_ORG_ID */
        10,                                        /* TSMPROLE_IS_NUMBER */
        'ZZZZZZZZ',                                /* TSMPROLE_ORG_ID */
        USER,                                      /* D_USERID_CODE */
        SYSDATE                                    /* D_LAST_UPDATE_TS */      
        );

    END IF;
  CLOSE c_check_for_person_role;

  /* make sure project, person assignment has not already been made */
  OPEN c_check_for_person(p_person_is_nbr, p_person_org_id, p_project_is_nbr, p_org_id);
  FETCH c_check_for_person INTO v_dummy;
    IF c_check_for_person%NOTFOUND THEN
      
      /* if nothing found then create association */
      INSERT INTO TSMPPA (
        TSMPROJ_IS_NUMBER,
        TSMPROJ_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS        
        )
      VALUES (
        p_project_is_nbr,                          /* TSMPROJ_IS_NUMBER */
        p_org_id,                                  /* TSMPROJ_ORG_ID */
        p_person_is_nbr,                           /* TSMPERSN_IS_NUMBER */
        p_person_org_id,                           /* TSMPERSN_ORG_ID */
        USER,                                      /* D_USERID_CODE */
        SYSDATE                                    /* D_LAST_UPDATE_TS */      
        );

    END IF;
  CLOSE c_check_for_person;  
  
  /* make sure role, person assignment has not already been made */
  OPEN c_check_for_role(p_person_is_nbr, p_person_org_id);
  FETCH c_check_for_role INTO v_dummy;
    IF c_check_for_role%NOTFOUND THEN
      
      /* if nothing found then create association */
      INSERT INTO TSMPRA (
        TSMPROLE_IS_NUMBER,
        TSMPROLE_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS        
        )
      VALUES (
        10,                                        /* TSMPROLE_IS_NUMBER */
        'ZZZZZZZZ',                                /* TSMPROLE_ORG_ID */
        p_person_is_nbr,                           /* TSMPERSN_IS_NUMBER */
        p_person_org_id,                           /* TSMPERSN_ORG_ID */
        USER,                                      /* D_USERID_CODE */
        SYSDATE                                    /* D_LAST_UPDATE_TS */      
        );

    END IF;
  CLOSE c_check_for_role;  


  COMMIT;
  
  RETURN 1;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in add person to project.');      
    RETURN 0;

END add_person_to_project;
                              
/******************************************************************************************************/
FUNCTION create_trip(p_trip_rec IN TSRTRIP%ROWTYPE) RETURN NUMBER IS

  
BEGIN
        
  /* create trip record */
  INSERT INTO TSRTRIP (
    TSRTRIP_IS_NUMBER,
    TSRTRIP_ORG_ID,
    ID_CODE,
    NAME,
    START_DATE,
    START_TIME,
    START_TIME_ZONE,
    END_DATE,
    END_TIME,
    END_TIME_ZONE,
    LEADER_NAME,
    VEHICLE_SHIP_NAME,
    TRIP_PLAN_TEXT,
    COMMENT_TEXT,
    SINGLE_PROJ_IND,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSMORGAN_IS_NUMBER
    ) 
  VALUES(
    p_trip_rec.TSRTRIP_IS_NUMBER,    /* TSRTRIP_IS_NUMBER */
    p_trip_rec.TSRTRIP_ORG_ID,       /* TSRTRIP_ORG_ID */
    p_trip_rec.ID_CODE,              /* ID_CODE */
    p_trip_rec.NAME,                 /* NAME */
    p_trip_rec.START_DATE,           /* START_DATE */
    p_trip_rec.START_TIME,           /* START_TIME */
    p_trip_rec.START_TIME_ZONE,      /* START_TIME_ZONE */
    p_trip_rec.END_DATE,             /* END_DATE */
    p_trip_rec.END_TIME,             /* END_TIME */
    p_trip_rec.END_TIME_ZONE,        /* END_TIME_ZONE */
    p_trip_rec.LEADER_NAME,          /* LEADER_NAME */
    p_trip_rec.VEHICLE_SHIP_NAME,    /* VEHICLE_SHIP_NAME */
    p_trip_rec.TRIP_PLAN_TEXT,       /* TRIP_PLAN_TEXT */
    p_trip_rec.COMMENT_TEXT,         /* COMMENT_TEXT */
    p_trip_rec.SINGLE_PROJ_IND,      /* SINGLE_PROJ_IND */
    p_trip_rec.D_USERID_CODE,        /* D_USERID_CODE */
    p_trip_rec.D_LAST_UPDATE_TS,     /* D_LAST_UPDATE_TS */
    p_trip_rec.TSMORGAN_IS_NUMBER    /* TSMORGAN_IS_NUMBER */
    );
            
  COMMIT;

  IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
    sim2_export_pkg.UPDATE_TSMEISN(rpad(p_trip_rec.TSRTRIP_ORG_ID,8),'TSRTRIP ', p_trip_rec.TSRTRIP_IS_NUMBER);
    COMMIT;
  END IF;
  
  
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create activity location.');     
    RETURN 0;  
  
END create_trip;

/******************************************************************************************************/
FUNCTION create_activity_location(p_org_id IN VARCHAR2,p_activity_is_nbr IN NUMBER,
                                  p_station_is_nbr IN NUMBER, p_station_org_id IN VARCHAR2) 
                                  RETURN NUMBER IS

  v_tsmalp_is_number             NUMBER(8);
  
  CURSOR c_abs_loc_point (p_station_is_nbr NUMBER, p_station_org_id VARCHAR2) IS
    SELECT TSMALP_IS_NUMBER FROM TSMALP WHERE TSMSTATN0IS_NUMBER = p_station_is_nbr
      AND TSMSTATN0ORG_ID = p_station_org_id;  

BEGIN
        
  /* get the malp is number via the station */
  OPEN c_abs_loc_point (p_station_is_nbr, p_station_org_id);
  FETCH c_abs_loc_point INTO v_tsmalp_is_number;
  CLOSE c_abs_loc_point;
          
  /* create activity location record */
  INSERT INTO TSRAAL (
    TSMALP_IS_NUMBER,
    TSMALP_ORG_ID,
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    TSRAAL_TYPE_NAME,
    D_USERID_CODE,
    D_LAST_UPDATE_TS
    ) 
  VALUES(
    v_tsmalp_is_number,         /*tsmalp_is_number*/
    RPAD(p_org_id,8),           /*tsmalp_org_id*/
    p_activity_is_nbr,          /*tsrfdact_is_number*/
    RPAD(p_org_id,8),           /*tsrfdact_org_id*/
    RPAD('General',15),         /* TSRAAL_TYPE_NAME */
    RPAD(USER,8),               /* D_USERID_CODE */
    SYSDATE                     /* D_LAST_UPDATE_TS */
    );
            
  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create activity location.');     
    RETURN 0;  
  
END create_activity_location;

/******************************************************************************************************/
FUNCTION create_station_visit(p_org_id IN VARCHAR2, p_stvst_rec IN TSRSTVST%ROWTYPE) 
  RETURN NUMBER IS

  
BEGIN

  BEGIN
    INSERT INTO TSRSTVST (
      TSRSTVST_IS_NUMBER,
      TSRSTVST_ORG_ID,     
      ID_NUMBER,      
      ARRIVAL_DATE,
      ARRIVAL_TIME,
      ARRIVAL_TIME_ZONE,
      DEPARTURE_DATE,
      DEPARTURE_TIME,
      DEPRTURE_TIME_ZONE,
      COMMENT_TEXT,
      D_USERID_CODE,
      D_LAST_UPDATE_TS,
      TSRTRIP_IS_NUMBER,
      TSRTRIP_ORG_ID,
      TSMSTATN_IS_NUMBER,
      TSMSTATN_ORG_ID
      )  
    VALUES (
      p_stvst_rec.tsrstvst_is_number     /* TSRSTVST_IS_NUMBER */,
      p_stvst_rec.tsrstvst_org_id        /* TSRSTVST_ORG_ID */,     
      p_stvst_rec.id_number              /* ID_NUMBER */,      
      p_stvst_rec.ARRIVAL_DATE           /* ARRIVAL_DATE */,
      p_stvst_rec.ARRIVAL_TIME           /* ARRIVAL_TIME */,
      p_stvst_rec.ARRIVAL_TIME_ZONE      /* ARRIVAL_TIME_ZONE */,
      p_stvst_rec.DEPARTURE_DATE         /* DEPARTURE_DATE */,
      p_stvst_rec.DEPARTURE_TIME         /* DEPARTURE_TIME */,
      p_stvst_rec.DEPRTURE_TIME_ZONE     /* DEPRTURE_TIME_ZONE */,
      p_stvst_rec.COMMENT_TEXT           /* COMMENT_TEXT */,
      RPAD(USER,8)                       /* D_USERID_CODE */,
      SYSDATE                            /* D_LAST_UPDATE_TS */,
      p_stvst_rec.TSRTRIP_IS_NUMBER      /* TSRTRIP_IS_NUMBER */,
      p_stvst_rec.TSRTRIP_ORG_ID         /* TSRTRIP_ORG_ID */,
      p_stvst_rec.TSMSTATN_IS_NUMBER     /* TSMSTATN_IS_NUMBER */,
      p_stvst_rec.TSMSTATN_ORG_ID        /* TSMSTATN_ORG_ID */   
      );

    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
      sim2_export_pkg.UPDATE_TSMEISN(rpad(p_org_id,8),'TSRSTVST', p_stvst_rec.tsrstvst_is_number);
      COMMIT;
    END IF;
 
--DBMS_OUTPUT.PUT_LINE('after station visit insert');

  EXCEPTION
    WHEN OTHERS THEN
      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  inserting visit record into TSRSTVST'); 
  END;
         
  RETURN 1;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create station visit.');   
     RETURN 0;
  
END create_station_visit;

/******************************************************************************************************/
FUNCTION create_activity(p_activity_rec IN OUT TSRFDACT%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create activity using info passed in through activity record */
  
  INSERT INTO TSRFDACT (
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    TYPE_NAME,
    CATEGORY_TYPE_NAME,
    ID_CODE,
    REPLICATE_NUMBER,
    MEDIUM_TYPE_NAME,
    INTENT_TYPE_NAME,
    COMMUNITY_NAME,
    START_DATE,
    START_TIME,
    START_TIME_ZONE,
    STOP_DATE,
    STOP_TIME,
    STOP_TIME_ZONE,
    COMMENT_TEXT,
    ZONE_TYPE_NAME,
    RELTV_DEPTH_NAME,
    DEPTH_REF_POINT,
    DEPTH_TO_ACTIVITY,
    DEPTH_TO_ACT_UN_CD,
    UPPER_DEPTH_TO_ACT,
    DEPTH_MSR_UNT_CD,
    LOWER_DEPTH_TO_ACT,
    THERMOCLINE_LOC_CD,
    PYCNOCLINE_LOC_CD,
    HALOCLINE_LOC_CD,
    SPECIES_NUMBER,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSROPPRD_ORG_ID,
    TSROPPRD_IS_NUMBER,
    TSRFDACT0IS_NUMBER,
    TSRFDACT0ORG_ID,
    TSRFDACT1IS_NUMBER,
    TSRFDACT1ORG_ID,
    TSRSTVST_IS_NUMBER,
    TSRSTVST_ORG_ID,
    TSRBIOPT_IS_NUMBER,
    TSRBIOPT_ORG_ID,
    TSRCHAR_IS_NUMBER,
    TSRCHAR_ORG_ID,
    TSRFLDPR_IS_NUMBER,
    TSRFLDPR_ORG_ID,
    TSRFLDGR_IS_NUMBER,
    TSRFLDGR_ORG_ID,
    TSRGRCFG_IS_NUMBER,
    TSRGRCFG_ORG_ID
    )
  VALUES (
    p_activity_rec.TSRFDACT_IS_NUMBER,          /* TSRFDACT_IS_NUMBER */
    p_activity_rec.TSRFDACT_ORG_ID,             /* TSRFDACT_ORG_ID */
    p_activity_rec.TYPE_NAME,                   /* TYPE_NAME */
    p_activity_rec.CATEGORY_TYPE_NAME,          /* CATEGORY_TYPE_NAME */
    p_activity_rec.ID_CODE,                     /* ID_CODE */
    p_activity_rec.REPLICATE_NUMBER,            /* REPLICATE_NUMBER */
    p_activity_rec.MEDIUM_TYPE_NAME,            /* MEDIUM_TYPE_NAME */
    p_activity_rec.INTENT_TYPE_NAME,            /* INTENT_TYPE_NAME */
    p_activity_rec.COMMUNITY_NAME,              /* COMMUNITY_NAME */
    p_activity_rec.START_DATE,                  /* START_DATE */
    p_activity_rec.START_TIME,                  /* START_TIME */
    p_activity_rec.START_TIME_ZONE,             /* START_TIME_ZONE */
    p_activity_rec.STOP_DATE,                   /* STOP_DATE */
    p_activity_rec.STOP_TIME,                   /* STOP_TIME */
    p_activity_rec.STOP_TIME_ZONE,              /* STOP_TIME_ZONE */
    p_activity_rec.COMMENT_TEXT,                /* COMMENT_TEXT */
    p_activity_rec.ZONE_TYPE_NAME,              /* ZONE_TYPE_NAME */
    p_activity_rec.RELTV_DEPTH_NAME,            /* RELTV_DEPTH_NAME */
    p_activity_rec.DEPTH_REF_POINT,             /* DEPTH_REF_POINT */
    p_activity_rec.DEPTH_TO_ACTIVITY,           /* DEPTH_TO_ACTIVITY */
    p_activity_rec.DEPTH_TO_ACT_UN_CD,          /* DEPTH_TO_ACT_UN_CD */
    p_activity_rec.UPPER_DEPTH_TO_ACT,          /* UPPER_DEPTH_TO_ACT */
    p_activity_rec.DEPTH_MSR_UNT_CD,            /* DEPTH_MSR_UNT_CD */
    p_activity_rec.LOWER_DEPTH_TO_ACT,          /* LOWER_DEPTH_TO_ACT */
    p_activity_rec.THERMOCLINE_LOC_CD,          /* THERMOCLINE_LOC_CD */
    p_activity_rec.PYCNOCLINE_LOC_CD,           /* PYCNOCLINE_LOC_CD */
    p_activity_rec.HALOCLINE_LOC_CD,            /* HALOCLINE_LOC_CD */
    p_activity_rec.SPECIES_NUMBER,              /* SPECIES_NUMBER */
    p_activity_rec.D_USERID_CODE,               /* D_USERID_CODE */
    p_activity_rec.D_LAST_UPDATE_TS,            /* D_LAST_UPDATE_TS */
    p_activity_rec.TSROPPRD_ORG_ID,             /* TSROPPRD_ORG_ID */
    p_activity_rec.TSROPPRD_IS_NUMBER,          /* TSROPPRD_IS_NUMBER */
    p_activity_rec.TSRFDACT0IS_NUMBER,          /* TSRFDACT0IS_NUMBER */
    p_activity_rec.TSRFDACT0ORG_ID,             /* TSRFDACT0ORG_ID */
    p_activity_rec.TSRFDACT1IS_NUMBER,          /* TSRFDACT1IS_NUMBER */
    p_activity_rec.TSRFDACT1ORG_ID,             /* TSRFDACT1ORG_ID */
    p_activity_rec.TSRSTVST_IS_NUMBER,          /* TSRSTVST_IS_NUMBER */
    p_activity_rec.TSRSTVST_ORG_ID,             /* TSRSTVST_ORG_ID */
    p_activity_rec.TSRBIOPT_IS_NUMBER,          /* TSRBIOPT_IS_NUMBER */
    p_activity_rec.TSRBIOPT_ORG_ID,             /* TSRBIOPT_ORG_ID */
    p_activity_rec.TSRCHAR_IS_NUMBER,           /* TSRCHAR_IS_NUMBER */
    p_activity_rec.TSRCHAR_ORG_ID,              /* TSRCHAR_ORG_ID */
    p_activity_rec.TSRFLDPR_IS_NUMBER,          /* TSRFLDPR_IS_NUMBER */
    p_activity_rec.TSRFLDPR_ORG_ID,             /* TSRFLDPR_ORG_ID */
    p_activity_rec.TSRFLDGR_IS_NUMBER,          /* TSRFLDGR_IS_NUMBER */
    p_activity_rec.TSRFLDGR_ORG_ID,             /* TSRFLDGR_ORG_ID */
    p_activity_rec.TSRGRCFG_IS_NUMBER,          /* TSRGRCFG_IS_NUMBER */
    p_activity_rec.TSRGRCFG_ORG_ID              /* TSRGRCFG_ORG_ID */
    );

    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
      sim2_export_pkg.UPDATE_TSMEISN( p_activity_rec.TSRFDACT_ORG_ID,'TSRFDACT', p_activity_rec.TSRFDACT_IS_NUMBER);
      COMMIT;
    END IF;
 
  RETURN 1;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create activity.');   
    RETURN 0;  

END create_activity;

/******************************************************************************************************/
FUNCTION create_sample(p_org_id IN VARCHAR2, p_sample_rec IN TSRSMPLE%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create a sample record */
  
  INSERT INTO TSRSMPLE (
    TSRFDACT_IS_NUMBER,            
    TSRFDACT_ORG_ID,               
    TOTAL_VOLUME_MSR,              
    TOTAL_VOLUME_UN_CD,            
    TOTAL_WEIGHT_MSR,              
    TOTAL_WEIGHT_UN_CD,            
    DATA_FILE_NAME_LOC,            
    CONTAINER_TYPE_NM,             
    CONTAINER_COLOR,               
    CONTAINER_SIZE_MSR,            
    CONTAINER_SIZE_UN,             
    TEMP_PRESERVN_TYPE,            
    PARENT_IND_CODE,               
    GEAR_DEPLOYMENT,               
    TRANSPORT_STORAGE,             
    PRESRV_STRGE_PRCDR,            
    DURATION_TIME,                 
    DURATION_UNITS,                
    D_USERID_CODE,                 
    D_LAST_UPDATE_TS
    )
  VALUES (
    p_sample_rec.TSRFDACT_IS_NUMBER,             /* TSRFDACT_IS_NUMBER */       
    p_sample_rec.TSRFDACT_ORG_ID,                /* TSRFDACT_ORG_ID */             
    p_sample_rec.TOTAL_VOLUME_MSR,               /* TOTAL_VOLUME_MSR */           
    p_sample_rec.TOTAL_VOLUME_UN_CD,             /* TOTAL_VOLUME_UN_CD */       
    p_sample_rec.TOTAL_WEIGHT_MSR,               /* TOTAL_WEIGHT_MSR */           
    p_sample_rec.TOTAL_WEIGHT_UN_CD,             /* TOTAL_WEIGHT_UN_CD */       
    p_sample_rec.DATA_FILE_NAME_LOC,             /* DATA_FILE_NAME_LOC */       
    p_sample_rec.CONTAINER_TYPE_NM,              /* CONTAINER_TYPE_NM */         
    p_sample_rec.CONTAINER_COLOR,                /* CONTAINER_COLOR */             
    p_sample_rec.CONTAINER_SIZE_MSR,             /* CONTAINER_SIZE_MSR */       
    p_sample_rec.CONTAINER_SIZE_UN,              /* CONTAINER_SIZE_UN */         
    p_sample_rec.TEMP_PRESERVN_TYPE,             /* TEMP_PRESERVN_TYPE */       
    p_sample_rec.PARENT_IND_CODE,                /* PARENT_IND_CODE */             
    p_sample_rec.GEAR_DEPLOYMENT,                /* GEAR_DEPLOYMENT */             
    p_sample_rec.TRANSPORT_STORAGE,              /* TRANSPORT_STORAGE */         
    p_sample_rec.PRESRV_STRGE_PRCDR,             /* PRESRV_STRGE_PRCDR */       
    p_sample_rec.DURATION_TIME,                  /* DURATION_TIME */                 
    p_sample_rec.DURATION_UNITS,                 /* DURATION_UNITS */               
    p_sample_rec.D_USERID_CODE,                  /* D_USERID_CODE */                 
    p_sample_rec.D_LAST_UPDATE_TS                /* D_LAST_UPDATE_TS */
    );           

  COMMIT;

  RETURN 1;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create sample.');   
    RETURN 0;  
  
END create_sample;
/******************************************************************************************************/
FUNCTION create_trip_qc(p_trip_qc_rec IN OUT TSRFQS%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create trip qc sample using info passed in through trip qc record */
  
  INSERT INTO TSRFQS (
    TSRFQS_IS_NUMBER,
    TSRFQS_ORG_ID,
    ID_CODE,
    TYPE_NAME,
    NAME,
    CREATION_DATE,
    CREATION_TIME,
    TIME_ZONE,
    TOTAL_VOLUME_MSR,
    TOTAL_VOLUME_UN_CD,
    CONTAINER_TYPE_NM,
    CONTAINER_COLOR,
    CONTAINER_SIZE_MSR,
    CONTAINER_SIZE_UN,
    COMMENT_TEXT,
    CREATION_METHOD,
    TRANSPORT_STORAGE,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSRTRIP_IS_NUMBER,
    TSRTRIP_ORG_ID
    )
  VALUES (
    p_trip_qc_rec.TSRFQS_IS_NUMBER,          /* TSRFQS_IS_NUMBER */
    p_trip_qc_rec.TSRFQS_ORG_ID,             /* TSRFQS_ORG_ID */
    p_trip_qc_rec.ID_CODE,                   /* ID_CODE */
    p_trip_qc_rec.TYPE_NAME,                 /* TYPE_NAME */
    p_trip_qc_rec.NAME,                      /* NAME */
    p_trip_qc_rec.CREATION_DATE,             /* CREATION_DATE */
    p_trip_qc_rec.CREATION_TIME,             /* CREATION_TIME */
    p_trip_qc_rec.TIME_ZONE,                 /* TIME_ZONE */
    p_trip_qc_rec.TOTAL_VOLUME_MSR,          /* TOTAL_VOLUME_MSR */
    p_trip_qc_rec.TOTAL_VOLUME_UN_CD,        /* TOTAL_VOLUME_UN_CD */
    p_trip_qc_rec.CONTAINER_TYPE_NM,         /* CONTAINER_TYPE_NM */
    p_trip_qc_rec.CONTAINER_COLOR,           /* CONTAINER_COLOR */
    p_trip_qc_rec.CONTAINER_SIZE_MSR,        /* CONTAINER_SIZE_MSR */
    p_trip_qc_rec.CONTAINER_SIZE_UN,         /* CONTAINER_SIZE_UN */
    p_trip_qc_rec.COMMENT_TEXT,              /* COMMENT_TEXT */
    p_trip_qc_rec.CREATION_METHOD,           /* CREATION_METHOD */
    p_trip_qc_rec.TRANSPORT_STORAGE,         /* TRANSPORT_STORAGE */
    p_trip_qc_rec.D_USERID_CODE,             /* D_USERID_CODE */
    p_trip_qc_rec.D_LAST_UPDATE_TS,          /* D_LAST_UPDATE_TS */
    p_trip_qc_rec.TSRTRIP_IS_NUMBER,         /* TSRTRIP_IS_NUMBER */
    p_trip_qc_rec.TSRTRIP_ORG_ID             /* TSRTRIP_ORG_ID */
    );

    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
      sim2_export_pkg.UPDATE_TSMEISN( p_trip_qc_rec.TSRFQS_ORG_ID,'TSRFQS', p_trip_qc_rec.TSRFQS_IS_NUMBER);
    END IF;

  RETURN 1;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create trip qc.');   
    RETURN 0;  

END create_trip_qc;

/******************************************************************************************************/
FUNCTION add_person_to_activity(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2, 
                                p_act_is_nbr IN NUMBER, p_act_org_id IN VARCHAR2) RETURN NUMBER IS

  v_persn_is_nbr             NUMBER(8);                                 
                                 
  CURSOR c_check_for_person(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2,
                            p_act_is_nbr IN NUMBER, p_act_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSRFAPEA WHERE TSRFDACT_IS_NUMBER = p_act_is_nbr
      AND TSMPERSN_IS_NUMBER = p_persn_is_nbr AND TSRFDACT_ORG_ID = p_act_org_id
      AND TSMPERSN_ORG_ID = p_persn_org_id;

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_person(p_persn_is_nbr,p_persn_org_id,p_act_is_nbr,p_act_org_id);
  FETCH c_check_for_person INTO v_persn_is_nbr;
    IF c_check_for_person%NOTFOUND THEN
      
      /* add person to activity if not already there */
      INSERT INTO TSRFAPEA (
        TSRFDACT_IS_NUMBER,
        TSRFDACT_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        p_act_is_nbr,                           /* TSRFDACT_IS_NUMBER */
        p_act_org_id,                           /* TSRFDACT_ORG_ID */
        p_persn_is_nbr,                         /* TSMPERSN_IS_NUMBER */
        p_persn_org_id,                         /* TSMPERSN_ORG_ID */
        USER,                                   /* D_USERID_CODE */
        SYSDATE                                 /* D_LAST_UPDATE_TS */     
        );                         
    
      COMMIT; 
    
    END IF;
  CLOSE c_check_for_person;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add person to activity.');   
    RETURN 0;
       
END add_person_to_activity;

/******************************************************************************************************/
FUNCTION add_person_to_trip_qc(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2, 
                               p_trip_qc_is_nbr IN NUMBER, p_trip_qc_org_id IN VARCHAR2) 
                               RETURN NUMBER IS

  v_persn_is_nbr             NUMBER(8);                                 
                                 
  CURSOR c_check_for_person(p_persn_is_nbr IN NUMBER, p_persn_org_id IN VARCHAR2,
                            p_trip_qc_is_nbr IN NUMBER, p_trip_qc_org_id IN VARCHAR2) IS
    SELECT TSMPERSN_IS_NUMBER FROM TSRFQSPA WHERE TSRFQS_IS_NUMBER = p_trip_qc_is_nbr
      AND TSMPERSN_IS_NUMBER = p_persn_is_nbr AND TSRFQS_ORG_ID = p_trip_qc_org_id
      AND TSMPERSN_ORG_ID = p_persn_org_id;

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_person(p_persn_is_nbr,p_persn_org_id,p_trip_qc_is_nbr,p_trip_qc_org_id);
  FETCH c_check_for_person INTO v_persn_is_nbr;
    IF c_check_for_person%NOTFOUND THEN
      
      /* add person to trip qc if not already there */
      INSERT INTO TSRFQSPA(
        TSRFQS_IS_NUMBER,
        TSRFQS_ORG_ID,
        TSMPERSN_IS_NUMBER,
        TSMPERSN_ORG_ID,
        D_USERID_CODE,
        D_LAST_UPDATE_TS
        )
      VALUES (
        p_trip_qc_is_nbr,                       /* TSRFQS_IS_NUMBER */
        p_trip_qc_org_id,                       /* TSRFQS_ORG_ID */
        p_persn_is_nbr,                         /* TSMPERSN_IS_NUMBER */
        p_persn_org_id,                         /* TSMPERSN_ORG_ID */
        USER,                                   /* D_USERID_CODE */
        SYSDATE                                 /* D_LAST_UPDATE_TS */     
        );                         
    
      COMMIT; 
    
    END IF;
  CLOSE c_check_for_person;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add person to trip qc.');   
    RETURN 0;
       
END add_person_to_trip_qc;

/******************************************************************************************************/
FUNCTION add_field_set_to_activity(p_act_is_nbr IN NUMBER, p_act_org IN VARCHAR2,
                                   p_fld_set_is_nbr IN NUMBER, p_fld_set_org IN VARCHAR2) 
                                   RETURN NUMBER IS
                                   
  v_fdset_is_nbr              NUMBER(8);                                   
                                   
  CURSOR c_check_for_set(p_act_is_nbr IN NUMBER, p_act_org IN VARCHAR2,
                         p_fld_set_is_nbr IN NUMBER, p_fld_set_org IN VARCHAR2) IS
    SELECT TSRFDSET_IS_NUMBER FROM TSRFAFSA WHERE TSRFDACT_IS_NUMBER = p_act_is_nbr
      AND TSRFDSET_IS_NUMBER = p_fld_set_is_nbr AND TSRFDACT_ORG_ID = RPAD(p_act_org,8)
      AND TSRFDSET_ORG_ID = RPAD(p_fld_set_org,8);

BEGIN

  /* make sure association isn't already there */
  OPEN c_check_for_set(p_act_is_nbr, p_act_org, p_fld_set_is_nbr, p_fld_set_org);
  FETCH c_check_for_set INTO v_fdset_is_nbr;
    IF c_check_for_set%NOTFOUND THEN
  
      /* if field set isn't on activity, add it */
      INSERT INTO TSRFAFSA (
        TSRFDACT_IS_NUMBER, 
        TSRFDACT_ORG_ID,
        TSRFDSET_IS_NUMBER, 
        TSRFDSET_ORG_ID,
        D_USERID_CODE, 
        D_LAST_UPDATE_TS
        ) 
      VALUES (
        p_act_is_nbr, 
        p_act_org,
        p_fld_set_is_nbr, 
        p_fld_set_org,
        RPAD(USER,8),
        SYSDATE
        );
       
      COMMIT;
      
    END IF;
  CLOSE c_check_for_set;
  
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in add field set to activity.');   
    RETURN 0;

END add_field_set_to_activity;

/******************************************************************************************************/
FUNCTION create_detection_record(p_detection_rec IN TSRDQL%ROWTYPE) RETURN NUMBER IS

BEGIN

  NULL;
  
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create detection record.');   
    RETURN 0;
  
END create_detection_record;

/******************************************************************************************************/
FUNCTION create_result(p_result_rec IN TSRRSULT%ROWTYPE) RETURN NUMBER IS

BEGIN

  /* create result using info from result record */
  INSERT INTO TSRRSULT (
    TSRRSULT_IS_NUMBER,
    TSRRSULT_ORG_ID,
    D_ASSND_SEQ_NUM,
    COMP_IND_CD,
    LAB_CERT_IND_CODE,
    VALUE_TEXT,
    VALUE_MEASURE,
    VALUE_STATUS,
    PRECISION_AMT_TEXT,
    LAB_BATCH_ID_CODE,
    DILUTION_IND_CODE,
    RECOVERY_IND_CODE,
    CORRECTION_IND_CD,
    RSLT_TXT_IND_CD,
    CELL_DESC_IND_CD,
    CONF_LVL_PCT_MSR,
    PROC_EXCPTN_IND_CD,
    DUR_BASIS_TYPE_NM,
    DETECT_COND_CD,
    SMPL_FRAC_TYPE_NM,
    STATISTIC_TYPE_NM,
    VALUE_TYPE_NAME,
    WT_BASIS_TYPE_NM,
    TEMP_BASIS_LVL_NM,
    SPECIES_NUMBER,
    REF_PT_FROM_NAME,
    REF_PT_TO_NAME,
    LOWER_RANGE_VALUE,
    UPPER_RANGE_VALUE,
    TAXON_POLLUTION,
    FNCTIONAL_FEED_GRP,
    TROPHIC_LEVEL,
    ANALYSIS_DATE,
    ANALYSIS_TIME,
    ANALYSIS_TIME_ZONE,
    REPL_ANALYSIS_NUM,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSRFDACT_IS_NUMBER,
    TSRFDACT_ORG_ID,
    TSRFQS_IS_NUMBER,
    TSRFQS_ORG_ID,
    TSRANLPR_IS_NUMBER,
    TSRANLPR_ORG_ID,
    TSRBRGI_IS_NUMBER,
    TSRBRGI_ORG_ID,
    TSRBRG_IS_NUMBER,
    TSRBRG_ORG_ID,
    TSRCHAR_IS_NUMBER,
    TSRCHAR_ORG_ID,
    TSRCHAR0IS_NUMBER,
    TSRCHAR0ORG_ID,
    TSRLAB_IS_NUMBER,
    TSRLAB_ORG_ID,
    TSRUOM_IS_NUMBER,
    TSRUOM_ORG_ID,
    TSRLSPP_IS_NUMBER,
    TSRLSPP_ORG_ID,
    TSRHCSC_IS_NUMBER,
    TSRHCSC_ORG_ID
    )
  VALUES (
    p_result_rec.TSRRSULT_IS_NUMBER,         /* TSRRSULT_IS_NUMBER */
    p_result_rec.TSRRSULT_ORG_ID,            /* TSRRSULT_ORG_ID */
    p_result_rec.D_ASSND_SEQ_NUM,            /* D_ASSND_SEQ_NUM */
    p_result_rec.COMP_IND_CD,                /* COMP_IND_CD */
    p_result_rec.LAB_CERT_IND_CODE,          /* LAB_CERT_IND_CODE */
    p_result_rec.VALUE_TEXT,                 /* VALUE_TEXT */
    p_result_rec.VALUE_MEASURE,              /* VALUE_MEASURE */
    p_result_rec.VALUE_STATUS,               /* VALUE_STATUS */
    p_result_rec.PRECISION_AMT_TEXT,         /* PRECISION_AMT_TEXT */
    p_result_rec.LAB_BATCH_ID_CODE,          /* LAB_BATCH_ID_CODE */
    p_result_rec.DILUTION_IND_CODE,          /* DILUTION_IND_CODE */
    p_result_rec.RECOVERY_IND_CODE,          /* RECOVERY_IND_CODE */
    p_result_rec.CORRECTION_IND_CD,          /* CORRECTION_IND_CD */
    p_result_rec.RSLT_TXT_IND_CD,            /* RSLT_TXT_IND_CD */
    p_result_rec.CELL_DESC_IND_CD,           /* CELL_DESC_IND_CD */
    p_result_rec.CONF_LVL_PCT_MSR,           /* CONF_LVL_PCT_MSR */
    p_result_rec.PROC_EXCPTN_IND_CD,         /* PROC_EXCPTN_IND_CD */
    p_result_rec.DUR_BASIS_TYPE_NM,          /* DUR_BASIS_TYPE_NM */
    p_result_rec.DETECT_COND_CD,             /* DETECT_COND_CD */
    p_result_rec.SMPL_FRAC_TYPE_NM,          /* SMPL_FRAC_TYPE_NM */
    p_result_rec.STATISTIC_TYPE_NM,          /* STATISTIC_TYPE_NM */
    p_result_rec.VALUE_TYPE_NAME,            /* VALUE_TYPE_NAME */
    p_result_rec.WT_BASIS_TYPE_NM,           /* WT_BASIS_TYPE_NM */
    p_result_rec.TEMP_BASIS_LVL_NM,          /* TEMP_BASIS_LVL_NM */
    p_result_rec.SPECIES_NUMBER,             /* SPECIES_NUMBER */
    p_result_rec.REF_PT_FROM_NAME,           /* REF_PT_FROM_NAME */
    p_result_rec.REF_PT_TO_NAME,             /* REF_PT_TO_NAME */
    p_result_rec.LOWER_RANGE_VALUE,          /* LOWER_RANGE_VALUE */
    p_result_rec.UPPER_RANGE_VALUE,          /* UPPER_RANGE_VALUE */
    p_result_rec.TAXON_POLLUTION,            /* TAXON_POLLUTION */
    p_result_rec.FNCTIONAL_FEED_GRP,         /* FNCTIONAL_FEED_GRP */
    p_result_rec.TROPHIC_LEVEL,              /* TROPHIC_LEVEL */
    p_result_rec.ANALYSIS_DATE,              /* ANALYSIS_DATE */
    p_result_rec.ANALYSIS_TIME,              /* ANALYSIS_TIME */
    p_result_rec.ANALYSIS_TIME_ZONE,         /* ANALYSIS_TIME_ZONE */
    p_result_rec.REPL_ANALYSIS_NUM,          /* REPL_ANALYSIS_NUM */
    p_result_rec.D_USERID_CODE,              /* D_USERID_CODE */
    p_result_rec.D_LAST_UPDATE_TS,           /* D_LAST_UPDATE_TS */
    p_result_rec.TSRFDACT_IS_NUMBER,         /* TSRFDACT_IS_NUMBER */
    p_result_rec.TSRFDACT_ORG_ID,            /* TSRFDACT_ORG_ID */
    p_result_rec.TSRFQS_IS_NUMBER,           /* TSRFQS_IS_NUMBER */
    p_result_rec.TSRFQS_ORG_ID,              /* TSRFQS_ORG_ID */
    p_result_rec.TSRANLPR_IS_NUMBER,         /* TSRANLPR_IS_NUMBER */
    p_result_rec.TSRANLPR_ORG_ID,            /* TSRANLPR_ORG_ID */
    p_result_rec.TSRBRGI_IS_NUMBER,          /* TSRBRGI_IS_NUMBER */
    p_result_rec.TSRBRGI_ORG_ID,             /* TSRBRGI_ORG_ID */
    p_result_rec.TSRBRG_IS_NUMBER,           /* TSRBRG_IS_NUMBER */
    p_result_rec.TSRBRG_ORG_ID,              /* TSRBRG_ORG_ID */
    p_result_rec.TSRCHAR_IS_NUMBER,          /* TSRCHAR_IS_NUMBER */
    p_result_rec.TSRCHAR_ORG_ID,             /* TSRCHAR_ORG_ID */
    p_result_rec.TSRCHAR0IS_NUMBER,          /* TSRCHAR0IS_NUMBER */
    p_result_rec.TSRCHAR0ORG_ID,             /* TSRCHAR0ORG_ID */
    p_result_rec.TSRLAB_IS_NUMBER,           /* TSRLAB_IS_NUMBER */
    p_result_rec.TSRLAB_ORG_ID,              /* TSRLAB_ORG_ID */
    p_result_rec.TSRUOM_IS_NUMBER,           /* TSRUOM_IS_NUMBER */
    p_result_rec.TSRUOM_ORG_ID,              /* TSRUOM_ORG_ID */
    p_result_rec.TSRLSPP_IS_NUMBER,          /* TSRLSPP_IS_NUMBER */
    p_result_rec.TSRLSPP_ORG_ID,             /* TSRLSPP_ORG_ID */
    p_result_rec.TSRHCSC_IS_NUMBER,          /* TSRHCSC_IS_NUMBER */
    p_result_rec.TSRHCSC_ORG_ID              /* TSRHCSC_ORG_ID */
    );

    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
      sim2_export_pkg.UPDATE_TSMEISN( p_result_rec.TSRRSULT_ORG_ID,'TSRRSULT', p_result_rec.TSRRSULT_IS_NUMBER);
    END IF;

  COMMIT;

  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in create result.');   
    RETURN 0;

END create_result;

/******************************************************************************************************/
PROCEDURE export_activity (p_activity_seq IN NUMBER, p_activity_rec IN OUT TSRFDACT%ROWTYPE, 
                           p_stvst_rec IN OUT TSRSTVST%ROWTYPE, p_import_seq IN NUMBER, 
                           p_org_id IN VARCHAR2) IS

   v_act_cat                       VARCHAR2(30);
   v_medium                        VARCHAR2(20);
   v_time_zone                     VARCHAR2(3);
   v_stop_time_zone                VARCHAR2(3);

   v_activity_is_nbr               NUMBER(8);
   v_tsranlpr_is_number            NUMBER(8);
   v_visit_is_number               NUMBER(8);
   v_field_set_is_number           NUMBER(8);
   v_lab_is_nbr                    NUMBER(8);
   v_lab_org_id                    VARCHAR2(8);
   v_anlpr_is_nbr                  NUMBER(8);
   v_anlpr_org_id                  VARCHAR2(8);
   
   v_rel_depth_code                VARCHAR2(30);
   v_depth_ref_point               VARCHAR2(30);
   v_depth_zone                    VARCHAR2(15);
   v_depth_unit                    VARCHAR2(3);
   v_depth_range_unit              VARCHAR2(3);
   v_thermo                        VARCHAR2(1);
   v_halo                          VARCHAR2(1);
   v_pycno                         VARCHAR2(1);
   v_sample_rec                    TSRSMPLE%ROWTYPE;
   v_sample_copy_rec               TSRSMPLE%ROWTYPE;
   v_trip_rec                      TSRTRIP%ROWTYPE;
   v_trip_copy_rec                 TSRTRIP%ROWTYPE;
   v_visit_rec                     TSRSTVST%ROWTYPE;
   v_visit_copy_rec                TSRSTVST%ROWTYPE;
   v_visit_id                      VARCHAR2(8);
   v_dummy_seq                     NUMBER(8);
   
   v_instr_choice                  VARCHAR2(15);
   v_station_id                    VARCHAR2(15);
   v_act_result_cnt                NUMBER(5);

   v_success                       NUMBER(1);      

   e_no_stvst                      EXCEPTION;
   
   CURSOR c_activity (p_org_id IN VARCHAR2, p_activity_seq IN NUMBER) IS 
      SELECT  
         FA_UPDEPTH, FA_LOWDEPTH, FA_DEPTHRU_SEQ,
         FA_TYPE_SEQ, FA_CATEGORY_SEQ, FA_ACTIVITY_ID, FA_START_DATE, FA_START_TIME, 
         FA_STOP_DATE, FA_STOP_TIME, FA_TIME_ZONE_SEQ, FA_DEPTH, FA_DEPTH_REF_POINT_SEQ,
         FA_THERMO_SEQ, FA_HALO_SEQ, FA_PYCNO_SEQ, FA_DEPTHZONE_SEQ,
         FA_REL_DEPTH_SEQ, FA_ACTIVITY_COMMENTS, FA_TSMSTATN_IS_NUMBER, FA_TSMSTATN_ORG_ID,
         FA_TSRTRIP_IS_NUMBER, FA_TSRTRIP_ORG_ID, FA_STP_SEQ, FA_TSRSTVST_IS_NUMBER,
         FA_TSRSTVST_ORG_ID, FA_SSV_SEQ, FA_TSRFLDPR_IS_NUMBER, FA_TSRFLDPR_ORG_ID, 
         FA_TSRFLDGR_IS_NUMBER, FA_TSRFLDGR_ORG_ID, FA_FDSET_ID, FA_FDSET_NAME, 
         FA_TSRLAB_IS_NUMBER, FA_TSRLAB_ORG_ID, FA_IMPORT_SEQ, FA_EXPORT_SEQ, FA_STATUS, 
         FA_TSMORGAN_ORG_ID, FA_APPROVED_BY, FA_APPROVED_ON, FA_TSRGRCFG_IS_NUMBER, 
         FA_TSRGRCFG_ORG_ID, FA_TSRSDP_IS_NUMBER, FA_TSRSDP_ORG_ID,
         TSRTRIP.ID_CODE TRIP_ID, FA_STVST_COMMENTS, FA_DU_SEQ, FA_MED_SEQ, FA_REPLICATE_NBR,
         TSRFLDPR.ID_CODE PR_ID_CODE, TSRFLDGR.ID_CODE GR_ID_CODE
      FROM 
         SIM_FIELD_ACTIVITIES, TSRFLDPR, TSRFLDGR, TSRTRIP
      WHERE 
         FA_TSRTRIP_IS_NUMBER = TSRTRIP_IS_NUMBER (+) AND
         RPAD(FA_TSRTRIP_ORG_ID,8) = TSRTRIP_ORG_ID (+) AND
         RPAD(FA_TSMORGAN_ORG_ID,8) = RPAD(p_org_id,8) AND
         FA_TSRFLDPR_IS_NUMBER = TSRFLDPR_IS_NUMBER(+) AND
         RPAD(FA_TSRFLDPR_ORG_ID,8) = TSRFLDPR_ORG_ID(+) AND
         FA_TSRFLDGR_IS_NUMBER = TSRFLDGR_IS_NUMBER(+) AND
         RPAD(FA_TSRFLDGR_ORG_ID,8) = TSRFLDGR_ORG_ID(+) AND
         /* kms FA_APPROVED_ON IS NOT NULL AND */
         FA_EXPORT_SEQ IS NULL AND
         FA_STATUS = 'A' AND
         FA_SEQ = p_activity_seq;
      
   CURSOR c_get_station_id(p_org_id IN VARCHAR2, p_station_is_nbr IN NUMBER) IS  
      SELECT IDENTIFICATION_CD FROM TSMSTATN WHERE TSMSTATN_ORG_ID = RPAD(p_org_id, 8)
        AND TSMSTATN_IS_NUMBER = p_station_is_nbr;
        
   CURSOR c_get_projects(p_activity_seq IN NUMBER) IS
     SELECT SAPJ_TSMPROJ_IS_NUMBER, SAPJ_TSMPROJ_ORG_ID FROM SIM_ACTIVITY_PROJECTS
       WHERE SAPJ_FA_SEQ = p_activity_seq;
        
   CURSOR c_get_personnel(p_activity_seq IN NUMBER) IS 
     SELECT SAP_TSMPERSN_IS_NUMBER, SAP_TSMPERSN_ORG_ID FROM SIM_ACTIVITY_PERSONNEL
       WHERE SAP_FA_SEQ = p_activity_seq;        
       
   CURSOR c_get_visit_info(p_visit_is_number IN NUMBER, p_visit_org_id IN VARCHAR2) IS
     SELECT TSRSTVST_IS_NUMBER FROM TSRSTVST T
       WHERE T.TSRSTVST_IS_NUMBER = p_visit_is_number
       AND   T.TSRSTVST_ORG_ID    = p_visit_org_id;
       
   CURSOR c_get_trip_info(p_trip_is_number IN NUMBER, p_trip_org_id IN VARCHAR2) IS
     SELECT TSRTRIP_IS_NUMBER FROM TSRTRIP T
       WHERE T.TSRTRIP_IS_NUMBER = p_trip_is_number
       AND   T.TSRTRIP_ORG_ID    = p_trip_org_id;

BEGIN

  FOR v_activity IN c_activity(p_org_id, p_activity_seq) LOOP

     /* get the allowable values */
     BEGIN     
       IF v_activity.fa_category_seq IS NOT NULL THEN
         v_act_cat := get_allowable_value(v_activity.fa_category_seq);
       ELSE
         v_act_cat := NULL;
       END IF;
       
       IF v_activity.fa_time_zone_seq IS NOT NULL THEN
         v_time_zone := get_allowable_value(v_activity.fa_time_zone_seq);
       ELSE
         v_time_zone := NULL;
       END IF;

       IF v_activity.fa_med_seq IS NOT NULL THEN
         v_medium := get_allowable_value(v_activity.fa_med_seq);
       ELSE
         v_medium := NULL;
       END IF;       
       
       IF v_activity.fa_rel_depth_seq IS NOT NULL THEN
         v_rel_depth_code := get_allowable_value(v_activity.fa_rel_depth_seq); 
       ELSE
         v_rel_depth_code := NULL;
       END IF;
       
       IF v_activity.fa_depthzone_seq IS NOT NULL THEN
         v_depth_zone := get_allowable_value(v_activity.fa_depthzone_seq);
       ELSE 
         v_depth_zone := NULL;
       END IF;
       
       IF v_activity.fa_thermo_seq IS NOT NULL THEN
         v_thermo := SUBSTR(get_allowable_value(v_activity.fa_thermo_seq),1,1);
       ELSE
         v_thermo := NULL;
       END IF;
       
       IF v_activity.fa_halo_seq IS NOT NULL THEN
         v_halo := SUBSTR(get_allowable_value(v_activity.fa_halo_seq),1,1);
       ELSE
         v_halo := NULL;
       END IF;
       
       IF v_activity.fa_pycno_seq IS NOT NULL THEN
         v_pycno := SUBSTR(get_allowable_value(v_activity.fa_pycno_seq),1,1);
       ELSE 
         v_pycno := NULL;
       END IF;
       
       IF v_activity.fa_start_time IS NULL THEN
         v_time_zone := NULL;
       END IF;

       IF v_activity.fa_stop_time IS NOT NULL THEN
         v_stop_time_zone := v_time_zone;
       ELSE
         v_stop_time_zone := NULL;
       END IF;

       IF v_activity.fa_du_seq IS NOT NULL THEN
         v_depth_unit := get_allowable_value(v_activity.fa_du_seq);
       ELSE
         v_depth_unit := NULL;
       END IF;

       IF v_activity.fa_depthru_seq IS NOT NULL THEN
         v_depth_range_unit := get_allowable_value(v_activity.fa_depthru_seq);     
       ELSE
         v_depth_range_unit := NULL;
       END IF;

     END;

     /* Verify that trip still exists in STORET */
     IF v_activity.FA_TSRTRIP_IS_NUMBER IS NOT NULL THEN
       
         OPEN c_get_trip_info(v_activity.FA_TSRTRIP_IS_NUMBER, v_activity.FA_TSRTRIP_ORG_ID);
         FETCH c_get_trip_info INTO v_dummy_seq;
               
           /* If trip does not still exist, clear is_number and let SIM recreate it. */
           IF c_get_trip_info%NOTFOUND THEN
             v_activity.FA_TSRTRIP_IS_NUMBER := NULL;
             v_activity.FA_TSRTRIP_ORG_ID    := NULL;
           END IF;
           
         CLOSE c_get_trip_info;
       
     END IF;      

     /* kms - if the trip does not already exist in STORET, get info needed to create it */
     IF v_activity.FA_TSRTRIP_IS_NUMBER IS NULL THEN
       get_trip_info(v_activity.FA_STP_SEQ,v_trip_rec);
       
       /* kms - call function to get next trip is number */
       v_activity.FA_TSRTRIP_IS_NUMBER := get_next_trip_is_nbr(p_org_id);
       
       v_trip_rec.TSRTRIP_IS_NUMBER := v_activity.FA_TSRTRIP_IS_NUMBER;
       v_trip_rec.TSRTRIP_ORG_ID := p_org_id;
       v_trip_rec.START_TIME_ZONE := NULL;  --v_time_zone; we don't import trip start time zone
       v_trip_rec.END_TIME_ZONE := NULL;    --v_stop_time_zone;
       v_trip_rec.SINGLE_PROJ_IND := 'Y';
       v_trip_rec.D_USERID_CODE := USER;
       v_trip_rec.D_LAST_UPDATE_TS := SYSDATE;
        
       /* kms - create the trip in STORET */
       v_success := create_trip(v_trip_rec);
              
       IF v_activity.FA_TSRTRIP_IS_NUMBER IS NOT NULL THEN
         v_activity.FA_TSRTRIP_ORG_ID := p_org_id;
         
         /* kms (6/6/01) - fill in the trip fk columns in the sim_trips record
            and in every activity with this trip. */
         UPDATE SIM_TRIPS 
           SET  STP_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
                STP_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE STP_SEQ = v_activity.FA_STP_SEQ;
           
         UPDATE SIM_FIELD_ACTIVITIES
           SET FA_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
               FA_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE FA_STP_SEQ =  v_activity.FA_STP_SEQ;
           
         COMMIT;
       END IF;
     END IF;
     
     OPEN c_get_station_id (p_org_id, v_activity.fa_tsmstatn_is_number);
     FETCH c_get_station_id INTO v_station_id;
     CLOSE c_get_station_id;
       
--DBMS_OUTPUT.PUT_LINE('After get station id');
          
     /* call function to get the next activity is number */
     v_activity_is_nbr := get_next_activity_is_nbr(p_org_id);

     v_depth_ref_point := v_activity.fa_depth_ref_point_seq;

     /* write the 'actual activity location' TSRAAL record */
     BEGIN

       v_success := create_activity_location(p_org_id,v_activity_is_nbr,v_activity.FA_TSMSTATN_IS_NUMBER, 
                                             v_activity.FA_TSMSTATN_ORG_ID);
                                                                                          
     EXCEPTION
       WHEN OTHERS THEN 
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - Writing tsraal record.');
     END;
  

--DBMS_OUTPUT.PUT_LINE('after tsraal');


     /* kms (6/4/01) - if station visit does not already exist, move what's
        in SIM into a new station visit record in STORET. */
     BEGIN
     
     
       /* Verify that visit still exists in STORET */
       IF v_activity.FA_TSRSTVST_IS_NUMBER IS NOT NULL THEN
         
           OPEN c_get_visit_info(v_activity.FA_TSRSTVST_IS_NUMBER, v_activity.FA_TSRSTVST_ORG_ID);
           FETCH c_get_visit_info INTO v_dummy_seq;
                 
             /* If visit does not still exist, clear is_number and let SIM recreate it. */
             IF c_get_visit_info%NOTFOUND THEN
               v_activity.FA_TSRSTVST_IS_NUMBER := NULL;
               v_activity.FA_TSRSTVST_ORG_ID    := NULL;
             END IF;
             
           CLOSE c_get_visit_info;
         
       END IF;      
       
       IF v_activity.FA_TSRSTVST_IS_NUMBER IS NULL THEN  
         /* kms - get info needed to create station visit */
         get_visit_info(v_activity.FA_SSV_SEQ,p_stvst_rec,v_visit_id);
         
         /* kms (6/8/01) - if the station visit id has not yet been assigned, use the 
            max id nbr used in STORET and increment it by 1. */
         IF v_visit_id LIKE 'UNA%' THEN 
           p_stvst_rec.ID_NUMBER := get_next_visit_id(v_activity.FA_TSRTRIP_IS_NUMBER,v_activity.FA_TSRTRIP_ORG_ID,
                                                      v_activity.FA_TSMSTATN_IS_NUMBER,v_activity.FA_TSMSTATN_ORG_ID);
         ELSE
           p_stvst_rec.ID_NUMBER := v_visit_id;
         END IF;
         
         /* kms - get the next station visit is number */
         v_visit_is_number := get_next_visit_is_number(p_org_id);
         p_stvst_rec.tsrstvst_is_number := v_visit_is_number;
         p_stvst_rec.tsrstvst_org_id := RPAD(p_org_id,8);
         
         /* kms - set remaining variables for station visit fields and create station visit in STORET. */
         --IF p_stvst_rec.ARRIVAL_TIME IS NOT NULL THEN
           --p_stvst_rec.ARRIVAL_TIME_ZONE := NVL(v_time_zone,'EST');
--         ELSE
           p_stvst_rec.ARRIVAL_TIME_ZONE := NULL;         
         --END IF;
         
         --IF p_stvst_rec.DEPARTURE_TIME IS NOT NULL THEN 
           --p_stvst_rec.DEPRTURE_TIME_ZONE := NVL(v_time_zone,'EST');
         --ELSE
           p_stvst_rec.DEPRTURE_TIME_ZONE := NULL;
         --END IF;
         
         p_stvst_rec.COMMENT_TEXT := v_activity.fa_stvst_comments;
         p_stvst_rec.TSRTRIP_IS_NUMBER := v_activity.fa_tsrtrip_is_number;
         p_stvst_rec.TSRTRIP_ORG_ID := v_activity.fa_tsrtrip_org_id;
         p_stvst_rec.TSMSTATN_IS_NUMBER := v_activity.fa_tsmstatn_is_number;
         p_stvst_rec.TSMSTATN_ORG_ID := v_activity.fa_tsmstatn_org_id;
         
         IF p_stvst_rec.ID_NUMBER IS NOT NULL THEN
           v_success := create_station_visit(p_org_id,p_stvst_rec); 
         ELSE
           sim2_export_pkg.write_error(v_export_log_seq, 'Did not get a station visit id number, unable to create station visit.');
           RAISE e_no_stvst;
         END IF;
 
         IF p_stvst_rec.tsrstvst_is_number IS NOT NULL THEN
           /* kms (6/6/01) - fill in the visit fk columns in the sim_trips record
              and in every activity with visit trip. */
   
           v_activity.FA_TSRSTVST_IS_NUMBER := p_stvst_rec.tsrstvst_is_number;
           v_activity.FA_TSRSTVST_ORG_ID := p_stvst_rec.tsrstvst_org_id;
 
           UPDATE SIM_STATION_VISITS 
             SET  SSV_TSRSTVST_IS_NUMBER = v_activity.FA_TSRSTVST_IS_NUMBER,
                  SSV_TSRSTVST_ORG_ID = v_activity.FA_TSRSTVST_ORG_ID,
                  SSV_ID = p_stvst_rec.ID_NUMBER
             WHERE SSV_SEQ = v_activity.FA_SSV_SEQ;
           
           UPDATE SIM_FIELD_ACTIVITIES
             SET FA_TSRSTVST_IS_NUMBER = v_activity.FA_TSRSTVST_IS_NUMBER,
                 FA_TSRSTVST_ORG_ID = v_activity.FA_TSRSTVST_ORG_ID
             WHERE FA_SSV_SEQ =  v_activity.FA_SSV_SEQ;
           
           COMMIT;
         ELSE
           RAISE e_no_stvst;
         END IF;
         
       ELSE
         v_visit_is_number := v_activity.FA_TSRSTVST_IS_NUMBER;
       END IF;


     EXCEPTION
       WHEN e_no_stvst THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Did not find or create station visit');       
     
       WHEN OTHERS THEN 
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing visit record for station');
     END;

     /* Add record to tsrtsa to associate station with trip */
     BEGIN
       add_station_to_trip(v_activity.FA_TSRTRIP_IS_NUMBER,v_activity.FA_TSMSTATN_IS_NUMBER,p_org_id);
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing station visited on trip record: '||v_activity.FA_TSMSTATN_IS_NUMBER);
     END;

--DBMS_OUTPUT.PUT_LINE('after add station to trip');
       
     BEGIN
        /* write the activity record */
    
        p_activity_rec.TSRFDACT_IS_NUMBER := v_activity_is_nbr;
        p_activity_rec.TSRFDACT_ORG_ID := RPAD(p_org_id,8);
--kms        p_activity_rec.TYPE_NAME := RPAD(v_act_type,13);
        p_activity_rec.CATEGORY_TYPE_NAME := RPAD(v_act_cat,30);
        p_activity_rec.ID_CODE := RPAD(v_activity.fa_activity_id,12);
        p_activity_rec.REPLICATE_NUMBER := v_activity.fa_replicate_nbr;
        p_activity_rec.MEDIUM_TYPE_NAME := RPAD(v_medium,20);
        p_activity_rec.INTENT_TYPE_NAME := NULL;
        p_activity_rec.COMMUNITY_NAME := NULL;
        p_activity_rec.START_DATE := TRUNC(v_activity.fa_start_date);
        p_activity_rec.START_TIME := v_activity.fa_start_time;
        p_activity_rec.START_TIME_ZONE := v_time_zone;
        p_activity_rec.STOP_DATE := v_activity.fa_stop_date;
        p_activity_rec.STOP_TIME := v_activity.fa_stop_time;
        p_activity_rec.STOP_TIME_ZONE := v_stop_time_zone;
        p_activity_rec.COMMENT_TEXT := v_activity.fa_activity_comments;
        p_activity_rec.ZONE_TYPE_NAME := v_depth_zone;
        p_activity_rec.RELTV_DEPTH_NAME := RPAD(v_rel_depth_code,15);
        p_activity_rec.DEPTH_REF_POINT := v_depth_ref_point;
        p_activity_rec.DEPTH_TO_ACTIVITY := v_activity.fa_depth;
        p_activity_rec.DEPTH_TO_ACT_UN_CD := RPAD(v_depth_unit,3);
        p_activity_rec.UPPER_DEPTH_TO_ACT := v_activity.fa_updepth;
        p_activity_rec.DEPTH_MSR_UNT_CD := RPAD(v_depth_range_unit,3);
        p_activity_rec.LOWER_DEPTH_TO_ACT := v_activity.fa_lowdepth;
        p_activity_rec.THERMOCLINE_LOC_CD := v_thermo;
        p_activity_rec.PYCNOCLINE_LOC_CD := v_pycno;
        p_activity_rec.HALOCLINE_LOC_CD := v_halo;
        p_activity_rec.SPECIES_NUMBER := NULL;
        p_activity_rec.D_USERID_CODE := USER;
        p_activity_rec.D_LAST_UPDATE_TS := SYSDATE;
        p_activity_rec.TSROPPRD_ORG_ID := NULL;
        p_activity_rec.TSROPPRD_IS_NUMBER := NULL;
        p_activity_rec.TSRFDACT0IS_NUMBER := NULL;
        p_activity_rec.TSRFDACT0ORG_ID := NULL;
        p_activity_rec.TSRFDACT1IS_NUMBER := NULL;
        p_activity_rec.TSRFDACT1ORG_ID := NULL;
        p_activity_rec.TSRSTVST_IS_NUMBER := v_visit_is_number;
        p_activity_rec.TSRSTVST_ORG_ID := RPAD(p_org_id,8);
        p_activity_rec.TSRBIOPT_IS_NUMBER := NULL;
        p_activity_rec.TSRBIOPT_ORG_ID := NULL;
        p_activity_rec.TSRCHAR_IS_NUMBER := NULL;
        p_activity_rec.TSRCHAR_ORG_ID := NULL;
        p_activity_rec.TSRFLDPR_IS_NUMBER := v_activity.FA_TSRFLDPR_IS_NUMBER;
        p_activity_rec.TSRFLDPR_ORG_ID := v_activity.FA_TSRFLDPR_ORG_ID;
        p_activity_rec.TSRFLDGR_IS_NUMBER := v_activity.FA_TSRFLDGR_IS_NUMBER;
        p_activity_rec.TSRFLDGR_ORG_ID := v_activity.FA_TSRFLDGR_ORG_ID;
        p_activity_rec.TSRGRCFG_IS_NUMBER := v_activity.FA_TSRGRCFG_IS_NUMBER;
        p_activity_rec.TSRGRCFG_ORG_ID := v_activity.FA_TSRGRCFG_ORG_ID;
       

        IF v_visit_is_number IS NOT NULL THEN
          v_success := create_activity(p_activity_rec);
        END IF;

--DBMS_OUTPUT.PUT_LINE('after field activities');
        /* if activity has the type sample, create record in the sample table */
        IF p_activity_rec.TYPE_NAME = RPAD('Sample',13) THEN
        
          /* kms (5/22/01)- get sample preservation, storage, and transport information */
          IF v_activity.FA_TSRSDP_IS_NUMBER IS NOT NULL THEN
            v_sample_rec := get_sample_pres_info(v_activity.FA_TSRSDP_IS_NUMBER,v_activity.FA_TSRSDP_ORG_ID,v_activity.fa_activity_id);
          END IF;
          
          v_sample_rec.TSRFDACT_IS_NUMBER := v_activity_is_nbr;            
          v_sample_rec.TSRFDACT_ORG_ID := RPAD(p_org_id,8);               
          v_sample_rec.TOTAL_VOLUME_MSR := NULL;              
          v_sample_rec.TOTAL_VOLUME_UN_CD := NULL;            
          v_sample_rec.TOTAL_WEIGHT_MSR := NULL;              
          v_sample_rec.TOTAL_WEIGHT_UN_CD := NULL;           
          v_sample_rec.DATA_FILE_NAME_LOC := NULL;         
          v_sample_rec.PARENT_IND_CODE := NULL;         
          v_sample_rec.GEAR_DEPLOYMENT := NULL;            
          v_sample_rec.TRANSPORT_STORAGE := NULL;            
          v_sample_rec.DURATION_TIME := NULL;          
          v_sample_rec.DURATION_UNITS := NULL;              
          v_sample_rec.D_USERID_CODE := USER;                 
          v_sample_rec.D_LAST_UPDATE_TS := SYSDATE;        
        
          v_success := create_sample(p_org_id, v_sample_rec);
        END IF;

     EXCEPTION
        WHEN OTHERS THEN 
           sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing activity ');
     END;

--both but different
     /* loop through projects, see if project is on the trip and activity, and add if it's not */
     BEGIN
       FOR v_project IN c_get_projects(p_activity_seq) LOOP
         add_project_to_trip(v_activity.FA_TSRTRIP_IS_NUMBER,v_project.SAPJ_TSMPROJ_IS_NUMBER,p_org_id);
         add_project_to_activity(v_activity_is_nbr,v_project.SAPJ_TSMPROJ_IS_NUMBER,p_org_id);
         
         /* make sure station is on project, add if it isn't */
         v_success:= add_station_to_project(v_project.SAPJ_TSMPROJ_IS_NUMBER,
                                            v_activity.FA_TSMSTATN_IS_NUMBER,p_org_id); 
                                            
         /* loop through personnel, assign to role and project*/
         BEGIN
           FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

             v_success := add_person_to_project(v_personnel.sap_tsmpersn_is_number,RPAD(v_personnel.sap_tsmpersn_org_id,8),
                                                v_project.SAPJ_TSMPROJ_IS_NUMBER,RPAD(p_org_id,8));
             
           END LOOP;
  
         EXCEPTION
           WHEN OTHERS THEN
             sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to project.');
         END;
                  
       END LOOP;
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding project to trip. Trip_is_nbr: '||v_activity.FA_TSRTRIP_IS_NUMBER);            
     END;                

--DBMS_OUTPUT.PUT_LINE('after add project to trip');      

--both but different
     /* loop through personnel, add to the activity */
     BEGIN
       FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

         v_success := add_person_to_activity(v_personnel.sap_tsmpersn_is_number,RPAD(v_personnel.sap_tsmpersn_org_id,8),
                                             v_activity_is_nbr, RPAD(p_org_id,8));

       END LOOP;
  
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to activity.');
     END;

     /* get the field set is number from STORET if exists or create if not */
     IF v_activity.fa_fdset_id IS NOT NULL THEN
       v_field_set_is_number := GET_FIELD_SET_IS_NBR(P_org_id, v_visit_is_number,
                                                     v_activity.fa_fdset_id, v_activity.fa_fdset_name);

       BEGIN
         /* associate activity with field set */        
         v_success := add_field_set_to_activity(v_activity_is_nbr,RPAD(p_org_id,8),v_field_set_is_number,
                                              RPAD(p_org_id,8));     
       EXCEPTION
         WHEN OTHERS THEN
           sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding field set to activity.');
       END;
     ELSE 
       v_field_set_is_number := NULL;
     END IF;
                                                   
     /* IF QC RESULTS, THERE WON'T BE AN ACTIVITY IS NUBR ... */
     BEGIN
       EXPORT_RESULTS  (p_activity_seq , p_activity_rec.type_name, v_activity_is_nbr, 
                        p_org_id,null,null,
                        v_activity.fa_tsrlab_is_number, v_activity.fa_tsrlab_org_id,
                        p_org_id , v_export_log_seq, v_act_result_cnt, v_time_zone);
     END;

     v_sample_rec := v_sample_copy_rec;
     v_trip_rec := v_trip_copy_rec;
     v_visit_rec := v_visit_copy_rec;
       
--DBMS_OUTPUT.PUT_LINE('after export results');
  
  END LOOP;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in export activity.');
                                                   
END export_activity;
/******************************************************************************************************/
PROCEDURE export_trip_qc (p_activity_seq IN NUMBER, p_trip_qc_rec IN OUT TSRFQS%ROWTYPE, 
                          p_import_seq IN NUMBER, p_org_id IN VARCHAR2) IS

   v_act_cat                       VARCHAR2(30);
   v_medium                        VARCHAR2(20);
   v_time_zone                     VARCHAR2(3);

   v_trip_qc_is_nbr                NUMBER(8);
   v_lab_is_nbr                    NUMBER(8);
   v_lab_org_id                    VARCHAR2(8);
   v_act_result_cnt                NUMBER(5);   
   
   v_trip_rec                      TSRTRIP%ROWTYPE;
   v_trip_copy_rec                 TSRTRIP%ROWTYPE;
   
   v_success                       NUMBER(1);      
   
   CURSOR c_activity (p_org_id IN VARCHAR2, p_activity_seq IN NUMBER) IS 
      SELECT  
         FA_CATEGORY_SEQ, FA_ACTIVITY_ID, 
         FA_START_DATE, FA_START_TIME, FA_TIME_ZONE_SEQ, FA_ACTIVITY_COMMENTS, 
         FA_TSRTRIP_IS_NUMBER, FA_TSRTRIP_ORG_ID, 
         FA_TSRLAB_IS_NUMBER, FA_TSRLAB_ORG_ID, FA_IMPORT_SEQ, FA_EXPORT_SEQ, FA_STATUS, 
         FA_TSMORGAN_ORG_ID, FA_APPROVED_BY, FA_APPROVED_ON, FA_MED_SEQ, FA_STP_SEQ
      FROM 
         SIM_FIELD_ACTIVITIES, TSRTRIP
      WHERE 
         FA_TSRTRIP_IS_NUMBER = TSRTRIP_IS_NUMBER (+) AND
         RPAD(FA_TSRTRIP_ORG_ID,8) = TSRTRIP_ORG_ID (+) AND
         RPAD(FA_TSMORGAN_ORG_ID,8) = RPAD(p_org_id,8) AND
         /* kms FA_APPROVED_ON IS NOT NULL AND */
         FA_EXPORT_SEQ IS NULL AND
         FA_STATUS = 'A' AND
         FA_SEQ = p_activity_seq;
        
   CURSOR c_get_projects(p_activity_seq IN NUMBER) IS
     SELECT SAPJ_TSMPROJ_IS_NUMBER, SAPJ_TSMPROJ_ORG_ID FROM SIM_ACTIVITY_PROJECTS
       WHERE SAPJ_FA_SEQ = p_activity_seq;
        
   CURSOR c_get_personnel(p_activity_seq IN NUMBER) IS 
     SELECT SAP_TSMPERSN_IS_NUMBER, SAP_TSMPERSN_ORG_ID FROM SIM_ACTIVITY_PERSONNEL
       WHERE SAP_FA_SEQ = p_activity_seq;        

BEGIN

  FOR v_activity IN c_activity(p_org_id, p_activity_seq) LOOP

     /* get the allowable values */
     BEGIN     
       IF v_activity.fa_category_seq IS NOT NULL THEN
         v_act_cat := get_allowable_value(v_activity.fa_category_seq);
       ELSE
         v_act_cat := NULL;
       END IF;
       
       IF v_activity.fa_time_zone_seq IS NOT NULL THEN
         v_time_zone := get_allowable_value(v_activity.fa_time_zone_seq);
       ELSE
         v_time_zone := NULL;
       END IF;

       IF v_activity.fa_med_seq IS NOT NULL THEN
         v_medium := get_allowable_value(v_activity.fa_med_seq);
       ELSE
         v_medium := NULL;
       END IF;       

     END;

     /* kms - if the trip does not already exist in STORET, get info needed to create it */
     IF v_activity.FA_TSRTRIP_IS_NUMBER IS NULL THEN
       get_trip_info(v_activity.FA_STP_SEQ,v_trip_rec);
       
       /* kms - call function to get next trip is number */
       v_activity.FA_TSRTRIP_IS_NUMBER := get_next_trip_is_nbr(p_org_id);

       v_trip_rec.TSRTRIP_IS_NUMBER := v_activity.FA_TSRTRIP_IS_NUMBER;
       v_trip_rec.TSRTRIP_ORG_ID := p_org_id;
       v_trip_rec.START_TIME_ZONE := NULL;  --v_time_zone;
       v_trip_rec.END_TIME_ZONE := NULL;    --v_time_zone;
       v_trip_rec.SINGLE_PROJ_IND := 'Y';
       v_trip_rec.D_USERID_CODE := USER;
       v_trip_rec.D_LAST_UPDATE_TS := SYSDATE;
       
       /* kms - create the trip in STORET */
       v_success := create_trip(v_trip_rec);
              
       IF v_activity.FA_TSRTRIP_IS_NUMBER IS NOT NULL THEN
         v_activity.FA_TSRTRIP_ORG_ID := p_org_id;
         
         /* kms (6/6/01) - fill in the trip fk columns in the sim_trips record
            and in every activity with this trip. */
         UPDATE SIM_TRIPS 
           SET  STP_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
                STP_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE STP_SEQ = v_activity.FA_STP_SEQ;
           
         UPDATE SIM_FIELD_ACTIVITIES
           SET FA_TSRTRIP_IS_NUMBER = v_activity.FA_TSRTRIP_IS_NUMBER,
               FA_TSRTRIP_ORG_ID = v_activity.FA_TSRTRIP_ORG_ID
           WHERE FA_STP_SEQ =  v_activity.FA_STP_SEQ;
           
         COMMIT;
         
       END IF;
     END IF;
     
     /* call function to get the next activity is number */
     v_trip_qc_is_nbr := get_next_trip_qc_is_nbr(p_org_id);

     BEGIN
     
       p_trip_qc_rec.TSRFQS_IS_NUMBER := v_trip_qc_is_nbr;
       p_trip_qc_rec.TSRFQS_ORG_ID := RPAD(p_org_id, 8);
       p_trip_qc_rec.ID_CODE := v_activity.fa_activity_id;
       p_trip_qc_rec.TYPE_NAME := v_act_cat;
       p_trip_qc_rec.NAME := NULL;
       p_trip_qc_rec.CREATION_DATE := v_activity.fa_start_date;
       p_trip_qc_rec.CREATION_TIME := v_activity.fa_start_time;
       p_trip_qc_rec.TIME_ZONE := v_time_zone;
       p_trip_qc_rec.TOTAL_VOLUME_MSR := NULL;
       p_trip_qc_rec.TOTAL_VOLUME_UN_CD := NULL;
       p_trip_qc_rec.CONTAINER_TYPE_NM := NULL;
       p_trip_qc_rec.CONTAINER_COLOR := NULL;
       p_trip_qc_rec.CONTAINER_SIZE_MSR := NULL;
       p_trip_qc_rec.CONTAINER_SIZE_UN := NULL;
       p_trip_qc_rec.COMMENT_TEXT := v_activity.fa_activity_comments;
       p_trip_qc_rec.CREATION_METHOD := NULL;
       p_trip_qc_rec.TRANSPORT_STORAGE := NULL;
       p_trip_qc_rec.D_USERID_CODE := USER;
       p_trip_qc_rec.D_LAST_UPDATE_TS := SYSDATE;
       p_trip_qc_rec.TSRTRIP_IS_NUMBER := v_activity.fa_tsrtrip_is_number;
       p_trip_qc_rec.TSRTRIP_ORG_ID := v_activity.fa_tsrtrip_org_id;       

       /* write the trip qc record */
       v_success := create_trip_qc(p_trip_qc_rec);

     EXCEPTION
        WHEN OTHERS THEN 
           sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  writing trip qc sample record.');
     END;

     /* loop through projects, see if project is on the trip, and add if it's not */
     BEGIN
       FOR v_project IN c_get_projects(p_activity_seq) LOOP
         add_project_to_trip(v_activity.FA_TSRTRIP_IS_NUMBER,v_project.SAPJ_TSMPROJ_IS_NUMBER,p_org_id);
         
         /* loop through personnel, assign to role and project*/
         BEGIN
           FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

             v_success := add_person_to_project(v_personnel.sap_tsmpersn_is_number,RPAD(v_personnel.sap_tsmpersn_org_id,8),
                                                v_project.SAPJ_TSMPROJ_IS_NUMBER,RPAD(p_org_id,8));
             
           END LOOP;
  
         EXCEPTION
           WHEN OTHERS THEN
             sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to project.');
         END;
                  
       END LOOP;
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding project to trip. Trip_is_nbr: '||v_activity.FA_TSRTRIP_IS_NUMBER);            
     END;                

--DBMS_OUTPUT.PUT_LINE('after add project to trip');      

--both but different
     /* loop through personnel, add to the sample */
     BEGIN
       FOR v_personnel IN c_get_personnel(p_activity_seq) LOOP

         v_success := add_person_to_trip_qc(v_personnel.sap_tsmpersn_is_number,RPAD(v_personnel.sap_tsmpersn_org_id,8),
                                            v_trip_qc_is_nbr, RPAD(p_org_id,8));

       END LOOP;
  
     EXCEPTION
       WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - adding person to trip qc sample.');
     END;

     /* IF QC RESULTS, THERE WON'T BE AN ACTIVITY IS NUBR ... */
     BEGIN
       EXPORT_RESULTS  (p_activity_seq , p_trip_qc_rec.type_name, null, null, 
                        v_trip_qc_is_nbr, p_org_id,
                        v_activity.fa_tsrlab_is_number, v_activity.fa_tsrlab_org_id,
                        p_org_id , v_export_log_seq,v_act_result_cnt, v_time_zone);
     END;
       
     v_trip_rec := v_trip_copy_rec;
--DBMS_OUTPUT.PUT_LINE('after export results');
  
  END LOOP;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in export trip qc.');
                                                   
END export_trip_qc;

/******************************************************************************************************/
PROCEDURE delete_exported_rec(p_fa_seq IN NUMBER) IS

BEGIN

  NULL;

END delete_exported_rec;

/******************************************************************************************************/
PROCEDURE SIM_EXPORT (p_org_id IN VARCHAR2, p_import_seq IN NUMBER) IS
   
   v_stvst_rec                     TSRSTVST%ROWTYPE;
   v_stvst_copy_rec                TSRSTVST%ROWTYPE;
   v_activity_rec                  TSRFDACT%ROWTYPE;
   v_activity_copy_rec             TSRFDACT%ROWTYPE;
   v_trip_qc_rec                   TSRFQS%ROWTYPE;
   v_trip_qc_copy_rec              TSRFQS%ROWTYPE;
   
   v_activity_is_nbr               NUMBER(8);
   v_activity_org_id               VARCHAR2(8);
   v_trip_qc_is_nbr                NUMBER(8);
   v_trip_qc_org_id                VARCHAR(8);
   v_visit_is_number               NUMBER(8);
   v_visit_org_id                  VARCHAR2(8);
   v_tsmalp_is_number              NUMBER(8);
   v_result_is_number              NUMBER(8) :=  NULL;
   v_tsranlpr_is_number            NUMBER(8);
   v_rel_depth_code                VARCHAR2(30);
   v_depth_ref_point               VARCHAR2(30);
   v_depth_zone                    VARCHAR2(15);
   v_depth_unit                    VARCHAR2(3);
   v_depth_range_unit              VARCHAR2(3);
   v_thermo                        VARCHAR2(1);
   v_halo                          VARCHAR2(1);
   v_pycno                         VARCHAR2(1);
   
   v_field_set_is_number           NUMBER(8);
   v_field_set_asgnmt_is_nbr       NUMBER(8);
   v_uom_is_nbr                    NUMBER(8);
   v_uom_org_id                    VARCHAR2(8);
   v_fldpr_is_nbr                  NUMBER(8);
   
   v_act_type                      VARCHAR2(13);
   v_act_cat                       VARCHAR2(30);
   v_medium                        VARCHAR2(20);
   v_time_zone                     VARCHAR2(3);
   v_stop_time_zone                VARCHAR2(3);
   
   v_result_value                  NUMBER(13,5);
   v_result_text                   VARCHAR2(15);
   v_min_detect                    VARCHAR2(12);
   v_max_detect                    VARCHAR2(12);
   v_detect_limit                  NUMBER(1);   
   v_detect_cond_cd                VARCHAR2(30);

   v_lab_is_nbr                    NUMBER(8);
   v_lab_org_id                    VARCHAR2(8);
   v_anlpr_is_nbr                  NUMBER(8);
   v_anlpr_org_id                  VARCHAR2(8);

   v_station_id                    VARCHAR2(15);
   v_trip_name                     VARCHAR2(15);

   v_sta_status                    VARCHAR2(1);
   v_value_status                  VARCHAR2(1);
   
   v_success                       NUMBER(1);   
   v_export_test                   NUMBER(10);
   
   ex_missing_info                 EXCEPTION;
   e_no_stvst                      EXCEPTION;

   CURSOR c_activity (p_org_id IN VARCHAR2, p_export_test IN NUMBER) IS 
      SELECT FA_SEQ, FA_TYPE_SEQ FROM SIM_FIELD_ACTIVITIES
        WHERE RPAD(FA_TSMORGAN_ORG_ID,8) = RPAD(p_org_id,8) 
        /* kms (6/27/01) we don't use this field right now -- AND FA_APPROVED_ON IS NOT NULL*/ 
        AND FA_EXPORT_SEQ IS NULL 
        AND FA_STATUS = 'A' AND FA_IMPORT_SEQ = p_import_seq
        ORDER BY FA_TSMSTATN_IS_NUMBER, FA_START_DATE, FA_START_TIME;
       
BEGIN
   /* kms - initialize activity and result counters */
   v_activity_cnt := 0;
   v_result_cnt := 0; 

   BEGIN
   /* set up export log */
      v_export_log_seq := sim2_export_pkg.GET_EXPORT_LOG_SEQ;

      
      INSERT INTO sim_export_log(
        SEL_SEQ,
        SEL_EXPORTED_BY,
        SEL_EXPORTED_ON,
        SEL_FILENAME,
        SEL_ORG_ID,
        SEL_EXPORT_TYPE
        )
      VALUES(
        v_export_log_seq,
        USER,
        SYSDATE,
        'Direct load: '||p_import_seq,
        p_org_id,
        'R'
        );

       COMMIT;
      
   EXCEPTION
      WHEN OTHERS THEN
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - setting up export log record.');
   END;

--DBMS_OUTPUT.PUT_LINE('After set up of export log');       

   sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Start: '||to_char(sysdate,'HH24:MI:SS'));

   /* loop through the activities */
   FOR v_activity IN c_activity(p_org_id, v_export_test) LOOP
   
--DBMS_OUTPUT.PUT_LINE('In activity loop');

     BEGIN
      
       /* get the activity type */
       IF v_activity.fa_type_seq IS NOT NULL THEN
         v_act_type := get_allowable_value(v_activity.fa_type_seq);
       ELSE
         v_act_type := NULL;
       END IF;
       
     EXCEPTION
       WHEN OTHERS THEN 
         sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  getting allowable values.');

     END; 
     
     /* see if this is a quality control sample or an activity */
     IF v_act_type != 'Trip QC' THEN
       /* set trip qc key to null */
       v_trip_qc_is_nbr := NULL;
       v_trip_qc_org_id := NULL;

       v_activity_rec.TYPE_NAME := v_act_type;

       /* run the procedure that writes activity to TSRFDACT */
       export_activity(v_activity.fa_seq, v_activity_rec, v_stvst_rec, p_import_seq, p_org_id);
       
       v_activity_is_nbr := v_activity_rec.TSRFDACT_IS_NUMBER;
       v_activity_org_id := v_activity_rec.TSRFDACT_ORG_ID;
       v_visit_is_number := v_stvst_rec.TSRSTVST_IS_NUMBER;
       v_visit_org_id := v_stvst_rec.TSRSTVST_ORG_ID;
       
     ELSE
       /* set field activity keys to null */
       v_activity_is_nbr := NULL;
       v_activity_org_id := NULL;
       v_visit_is_number := NULL;
       v_visit_org_id := NULL;

       v_trip_qc_rec.TYPE_NAME := v_act_type;       

       /* run the procedure that writes activity to TSRFQS */
       export_trip_qc (v_activity.fa_seq, v_trip_qc_rec, p_import_seq, p_org_id);

       v_trip_qc_is_nbr := v_trip_qc_rec.TSRFQS_IS_NUMBER;
       v_trip_qc_org_id := v_trip_qc_rec.TSRFQS_ORG_ID;

     END IF;

     v_activity_cnt := v_activity_cnt + 1;   
   
          
    /* update the activity as being exported */
    UPDATE SIM_FIELD_ACTIVITIES
    SET FA_EXPORT_SEQ = v_export_log_seq,
       FA_TSRFDACT_IS_NUMBER = v_activity_is_nbr,
       FA_TSRFDACT_ORG_ID    = RPAD(p_org_id,8),
       FA_TSRSTVST_IS_NUMBER = v_visit_is_number,
       FA_TSRSTVST_ORG_ID = RPAD(p_org_id,8),
       FA_TSRFQS_IS_NUMBER = v_trip_qc_is_nbr,
       FA_TSRFQS_ORG_ID = RPAD(p_org_id,8),
       FA_STATUS     = 'E'
    WHERE FA_SEQ = v_activity.fa_seq;

    COMMIT;

    v_stvst_rec := v_stvst_copy_rec;
    v_activity_rec := v_activity_copy_rec;
    
  END LOOP;
   
  /* update export log with count */
  UPDATE SIM_EXPORT_LOG SET SEL_ACTIVITY_CNT = v_activity_cnt 
  WHERE SEL_SEQ = v_export_log_seq;
  
  /* update the import log with export info 
     SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE SIM_IMPORT_LOG SET 
     SIL_EXPORTED_ON = SYSDATE,
     SIL_EXPORTED_BY = USER,
     SIL_ROWS_READY = SIL_ROWS_READY - v_result_cnt
  WHERE SIL_SEQ = P_IMPORT_SEQ;
  
  /* update counters */
  /* increment the is number in the TSMEISN table to the next one avail for manual entry */
  sim2_export_pkg.UPDATE_EISN_TABLE (v_cur_fieldset_is_nbr,v_cur_trip_is_nbr,v_cur_result_is_nbr,
                                     v_cur_tsmgntxt_is_nbr,v_cur_activity_is_nbr,v_cur_stavst_is_nbr,
                                     v_cur_trip_qc_is_nbr,p_org_id); 

  sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Exported '||v_activity_cnt||' activities, '||v_result_cnt||' results.');
  sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Finish: '||to_char(sysdate,'HH24:MI:SS'));

  COMMIT;

EXCEPTION
   WHEN OTHERS THEN
      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  Error in sim2_export_pkg.EXPORT_SIM');

END SIM_EXPORT;

/******************************************************************************************************/
PROCEDURE EXPORT_RESULTS  (p_fa_seq IN NUMBER, p_fat_desc IN VARCHAR2, p_activity_is_nbr IN NUMBER,
                           p_activity_org_id IN VARCHAR2, p_tripqc_is_nbr IN NUMBER, p_tripqc_org_id IN VARCHAR2,
                           p_lab_is_number IN NUMBER,p_lab_org_id IN VARCHAR2, p_org_id IN VARCHAR2, 
                           p_export_log_seq IN NUMBER, p_act_result_cnt IN OUT NUMBER, p_time_zone IN VARCHAR2) IS

  v_success                       NUMBER(1);
  v_activity_id                   VARCHAR2(12);
  v_result_rec                    TSRRSULT%ROWTYPE;
  v_result_copy_rec               TSRRSULT%ROWTYPE;

  v_result_is_number              NUMBER(8);
  v_lab_is_nbr                    NUMBER(8);
  v_lab_org_id                    VARCHAR2(8);

  v_result_value                  NUMBER(13,5);
  v_result_text                   VARCHAR2(15);
  v_value_status                  VARCHAR2(1);
  v_value_type                    VARCHAR2(10);
  v_stat_type                     VARCHAR2(18);
  v_dur_basis                     VARCHAR2(10);
  v_temp_basis                    VARCHAR2(8);
  v_conflevel                     VARCHAR2(8);
  v_det_limit                     VARCHAR2(12);
  v_min_detect                    VARCHAR2(12);
  v_max_detect                    VARCHAR2(12);
  v_detect_cond_cd                VARCHAR2(50);
  v_result_cmt                    VARCHAR2(1999);
  v_tsmgntxt_is_nbr               NUMBER(8);
  v_comp_ind_cd                   VARCHAR2(1);
  v_table                         VARCHAR2(8);
  v_desc                          VARCHAR2(8);
  
  e_def_not_complete              EXCEPTION;
  e_no_chgrp                      EXCEPTION;
  e_no_char                       EXCEPTION;
  e_no_rowid                      EXCEPTION;
  
  CURSOR c_result (p_activity_seq IN NUMBER) IS
    SELECT RS_SEQ, RS_AC_SEQ, RS_RESULT, RS_COMMENT, RS_VALUE_STATUS_SEQ, RS_LAB_BATCH_ID,
      RS_LAB_CERT, RS_DETECT_LIMIT, RS_DET_LIMIT_UOM_NBR, RS_DET_LIMIT_UOM_ORG, RS_ANALYSIS_DATE,
      RS_UPPER_QUANT_LIMIT, RS_LOWER_QUANT_LIMIT, RS_DETECT_COMMENT, RS_PRECISION, RS_CONFLEVEL_SEQ,
      RS_TSRCHAR_IS_NUMBER, RS_TSRCHAR_ORG_ID, RS_TSRUOM_IS_NBR, RS_TSRUOM_ORG_ID, RS_FRACTION,
      RS_VALUE_TYPE_SEQ, RS_STAT_TYPE_SEQ, RS_TSRANLPR_IS_NUMBER, RS_TSRANLPR_ORG_ID,
      RS_TSRLSPP_IS_NUMBER, RS_TSRLSPP_ORG_ID, RS_DURBASIS_SEQ, RS_TEMPBASIS_SEQ, RS_ANALYSIS_TIME,
      RS_TSRLAB_IS_NUMBER, RS_TSRLAB_ORG_ID
    FROM SIM_RESULTS 
      WHERE 
        RS_FA_SEQ = p_activity_seq;
    
  CURSOR c_get_activity_id(p_activity_seq IN NUMBER) IS 
    SELECT FA_ACTIVITY_ID FROM SIM_FIELD_ACTIVITIES WHERE FA_SEQ = p_activity_seq;
     
BEGIN

  p_act_result_cnt := 0;
     
  FOR v_result IN c_result (p_fa_seq) LOOP
    BEGIN
      /* initialize variables */
      v_result_value := NULL;
      v_result_text := NULL;
      v_value_status := NULL;
      v_value_type := NULL;
      v_stat_type := NULL;
      v_dur_basis := NULL;
      v_temp_basis := NULL;
      v_conflevel := NULL;
      v_det_limit := NULL;
      v_min_detect := NULL;
      v_max_detect := NULL;
      v_detect_cond_cd := NULL;
      v_result_cmt := NULL;
      v_comp_ind_cd := NULL;

      v_result_is_number := get_next_result_is_nbr (p_org_id);
      
      BEGIN
        /* value type */
        IF v_result.rs_value_type_seq IS NOT NULL THEN
          v_value_type := get_allowable_value(v_result.rs_value_type_seq);
        ELSE 
          v_value_type := NULL;
        END IF;
                
        /* stat type */
        IF v_result.rs_stat_type_seq IS NOT NULL THEN        
          v_stat_type := get_allowable_value(v_result.rs_stat_type_seq);        
        ELSE
          v_stat_type := NULL;
        END IF;
        
        /* duration basis */
        IF v_result.rs_durbasis_seq IS NOT NULL THEN
          v_dur_basis := get_allowable_value(v_result.rs_durbasis_seq);        
        ELSE 
          v_dur_basis := NULL;
        END IF;
        
        /* temp basis */
        IF v_result.rs_tempbasis_seq IS NOT NULL THEN
          v_temp_basis := get_allowable_value(v_result.rs_tempbasis_seq);                
        ELSE
          v_temp_basis := NULL;
        END IF;
        
        /* confidence level */
        IF v_result.rs_conflevel_seq IS NOT NULL THEN
          v_conflevel := get_allowable_value(v_result.rs_conflevel_seq);                
        ELSE
          v_conflevel := NULL;
        END IF;

      EXCEPTION
        WHEN OTHERS THEN
          sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting result characteristic information');
      END;
     
      IF p_fat_desc = 'Field Msr/Obs' THEN
        v_lab_is_nbr := NULL;
        v_lab_org_id := NULL;
      ELSE
        v_lab_is_nbr := v_result.rs_tsrlab_is_number;       --p_lab_is_number;
        v_lab_org_id := RPAD(v_result.rs_tsrlab_org_id, 8); --RPAD(p_lab_org_id,8);
      END IF;
       
      /* convert result text to number */
      v_result_text := v_result.rs_result;
      v_result_value := get_value_msr(v_result_text);

      /* get result status */         
      IF v_result.rs_value_status_seq IS NOT NULL THEN                             
        v_value_status := get_allowable_value(v_result.rs_value_status_seq);
      ELSE
        v_value_status := 'F';
      END IF;
      
      /* comp_ind_cd should always be 'Y' if the activity has been allowed to be exported */
      v_comp_ind_cd := 'Y';

      /* kms - set detection condition to correct value in order for the correct radio
         button to be selected in the STORET result screen. */
      IF v_result_text = '*Non-detect' THEN
        v_detect_cond_cd := 'Not Detected';
      ELSIF v_result_text = '*Present <QL' THEN
        v_detect_cond_cd := 'Present below Quantification';
      ELSIF v_result_text = '*Present >QL' THEN
        v_detect_cond_cd := 'Present above Quantification';
      ELSIF v_result_text = '*Present' THEN
        v_detect_cond_cd := 'Detected not Quantified';
      ELSE
        v_detect_cond_cd := 'Detected and Quantified';
      END IF;        
      
      /* write values to be exported to result record */      
      BEGIN
        v_result_rec.TSRRSULT_IS_NUMBER := v_result_is_number;
        v_result_rec.TSRRSULT_ORG_ID := RPAD(p_org_id,8);
        v_result_rec.D_ASSND_SEQ_NUM := 2;
        v_result_rec.COMP_IND_CD := v_comp_ind_cd;
        v_result_rec.LAB_CERT_IND_CODE := v_result.rs_lab_cert;
        v_result_rec.VALUE_TEXT := RPAD(v_result_text,15);
        v_result_rec.VALUE_MEASURE := v_result_value;
        v_result_rec.VALUE_STATUS := v_value_status;
        v_result_rec.PRECISION_AMT_TEXT := v_result.rs_precision;
        v_result_rec.LAB_BATCH_ID_CODE := v_result.rs_lab_batch_id;
        v_result_rec.DILUTION_IND_CODE := NULL;
        v_result_rec.RECOVERY_IND_CODE := NULL;
        v_result_rec.CORRECTION_IND_CD := NULL;
        v_result_rec.RSLT_TXT_IND_CD := NULL;
        v_result_rec.CELL_DESC_IND_CD := NULL;
        v_result_rec.CONF_LVL_PCT_MSR := v_conflevel;
        v_result_rec.PROC_EXCPTN_IND_CD := NULL;
        v_result_rec.DUR_BASIS_TYPE_NM := v_dur_basis;
        v_result_rec.DETECT_COND_CD := RPAD(v_detect_cond_cd,30);
        v_result_rec.SMPL_FRAC_TYPE_NM := v_result.rs_fraction;
        v_result_rec.STATISTIC_TYPE_NM := v_stat_type;
        v_result_rec.VALUE_TYPE_NAME := v_value_type;
        v_result_rec.WT_BASIS_TYPE_NM := NULL;
        v_result_rec.TEMP_BASIS_LVL_NM := v_temp_basis;
        v_result_rec.SPECIES_NUMBER := NULL;
        v_result_rec.REF_PT_FROM_NAME := NULL;
        v_result_rec.REF_PT_TO_NAME := NULL;
        v_result_rec.LOWER_RANGE_VALUE := NULL;--v_result.lower_range_value;
        v_result_rec.UPPER_RANGE_VALUE := NULL;--v_result.upper_range_value;
        v_result_rec.TAXON_POLLUTION := NULL;
        v_result_rec.FNCTIONAL_FEED_GRP := NULL;
        v_result_rec.TROPHIC_LEVEL := NULL;
        v_result_rec.ANALYSIS_DATE := v_result.rs_analysis_date;
        v_result_rec.ANALYSIS_TIME := v_result.rs_analysis_time;
        v_result_rec.ANALYSIS_TIME_ZONE := NULL;
        v_result_rec.REPL_ANALYSIS_NUM := NULL;
        v_result_rec.D_USERID_CODE := USER;
        v_result_rec.D_LAST_UPDATE_TS := SYSDATE;
        v_result_rec.TSRFDACT_IS_NUMBER := p_activity_is_nbr;
        v_result_rec.TSRFDACT_ORG_ID := RPAD(p_activity_org_id,8);
        v_result_rec.TSRFQS_IS_NUMBER := p_tripqc_is_nbr;
        v_result_rec.TSRFQS_ORG_ID := RPAD(p_tripqc_org_id,8);
        v_result_rec.TSRANLPR_IS_NUMBER := v_result.rs_tsranlpr_is_number;
        v_result_rec.TSRANLPR_ORG_ID := RPAD(v_result.rs_tsranlpr_org_id,8);
        v_result_rec.TSRBRGI_IS_NUMBER := NULL;
        v_result_rec.TSRBRGI_ORG_ID := NULL;
        v_result_rec.TSRBRG_IS_NUMBER := NULL;
        v_result_rec.TSRBRG_ORG_ID := NULL;
        v_result_rec.TSRCHAR_IS_NUMBER := v_result.rs_tsrchar_is_number;
        v_result_rec.TSRCHAR_ORG_ID := v_result.rs_tsrchar_org_id;
        v_result_rec.TSRCHAR0IS_NUMBER := NULL;
        v_result_rec.TSRCHAR0ORG_ID := NULL;
        v_result_rec.TSRLAB_IS_NUMBER := v_lab_is_nbr;
        v_result_rec.TSRLAB_ORG_ID := v_lab_org_id;
        v_result_rec.TSRUOM_IS_NUMBER := v_result.rs_tsruom_is_nbr;
        v_result_rec.TSRUOM_ORG_ID := RPAD(v_result.rs_tsruom_org_id,8);
        v_result_rec.TSRLSPP_IS_NUMBER := v_result.rs_tsrlspp_is_number;
        v_result_rec.TSRLSPP_ORG_ID := v_result.rs_tsrlspp_org_id;
        v_result_rec.TSRHCSC_IS_NUMBER := NULL;
        v_result_rec.TSRHCSC_ORG_ID := NULL;
      EXCEPTION
        WHEN OTHERS THEN
          sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - assigning values to result record.');
      END;
      
      /* create STORET result record */
      v_success := create_result(v_result_rec);
      v_result_cmt := v_result.RS_COMMENT;      -- dcl add to actually insert comment
         IF v_result.RS_COMMENT = '*Text' THEN     -- If the comment indicates it is a text 
            v_table := 'TSRRSULT';                 -- characteristic, insert data using the 
            v_desc  := 'DESCRIPT';                 -- correct table and value for description
         ELSE                                      -- name  
            v_table := 'TSRRSLT';
            v_desc  := 'RSLTCMNT';
         END IF;
        
      IF v_result_cmt IS NOT NULL THEN
        v_tsmgntxt_is_nbr := get_next_tsmgntxt_nbr(p_org_id);
        BEGIN
          INSERT INTO TSMGNTXT (
            TSMGNTXT_IS_NUMBER,
            TSMGNTXT_ORG_ID,
            TABLE_NAME,
            DESCRIPTION_NAME,
            DESCRIPTION_TEXT,
            D_USERID_CODE,
            D_LAST_UPDATE_TS,
            TSRRSULT_IS_NUMBER,
            TSRRSULT_ORG_ID,
            TSMPROJ_IS_NUMBER,
            TSMPROJ_ORG_ID,
            TSMPROGM_IS_NUMBER,
            TSMPROGM_ORG_ID,
            TSRGRCFG_IS_NUMBER,
            TSRGRCFG_ORG_ID  
            )
          VALUES (
            v_tsmgntxt_is_nbr,                                 /* TSMGNTXT_IS_NUMBER */
            RPAD(p_org_id,8),                                  /* TSMGNTXT_ORG_ID */
            v_table,                                           /* TABLE_NAME */
            v_desc,                                            /* DESCRIPTION_NAME */
            v_result_cmt,                                      /* DESCRIPTION_TEXT */
            USER,                                              /* D_USERID_CODE */
            SYSDATE,                                           /* D_LAST_UPDATE_TS */
            v_result_rec.TSRRSULT_IS_NUMBER,                   /* TSRRSULT_IS_NUMBER */
            v_result_rec.TSRRSULT_ORG_ID,                      /* TSRRSULT_ORG_ID */
            NULL,                                              /* TSMPROJ_IS_NUMBER */
            NULL,                                              /* TSMPROJ_ORG_ID */
            NULL,                                              /* TSMPROGM_IS_NUMBER */
            NULL,                                              /* TSMPROGM_ORG_ID */
            NULL,                                              /* TSRGRCFG_IS_NUMBER */
            NULL                                               /* TSRGRCFG_ORG_ID */
            );
      
          IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
            sim2_export_pkg.UPDATE_TSMEISN(rpad(p_org_id,8),'TSMGNTXT', v_tsmgntxt_is_nbr);
            COMMIT;
          END IF;
 
        EXCEPTION
          WHEN OTHERS THEN
            sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - writing result comment.');
        END;
      END IF;

      /* figure out detect limit info */
      BEGIN
        v_result_text := v_result.RS_RESULT;
        v_det_limit := v_result.RS_DETECT_LIMIT;
        v_result_cmt := v_result.RS_COMMENT;
        v_min_detect := v_result.RS_LOWER_QUANT_LIMIT;
        v_max_detect := v_result.RS_UPPER_QUANT_LIMIT;
        /*
        IF v_det_limit IS NULL AND (v_min_detect IS NOT NULL OR v_max_detect IS NOT NULL) THEN
          /* kms (6/13/01) - if this dql record is for quantification limits, the units from the 
             result need to be put in as the units rather than the detection limit units. */ /*
          v_result.rs_det_limit_uom_nbr := v_result.rs_tsruom_is_nbr;
          v_result.rs_det_limit_uom_nbr := v_result.rs_tsruom_org_id;
        END IF;
        */   
        IF v_det_limit IS NOT NULL OR v_min_detect IS NOT NULL OR v_max_detect IS NOT NULL THEN
          BEGIN
            INSERT INTO TSRDQL (
              TSRRSULT_IS_NUMBER,
              TSRRSULT_ORG_ID,
              MIN_DETECT_LIMIT,
              MIN_QUANT_LIMIT,
              MAX_QUANT_LIMIT,
              DESCRIPTION_TEXT,
              D_USERID_CODE,
              D_LAST_UPDATE_TS,
              TSRUOM_IS_NUMBER,
              TSRUOM_ORG_ID) 
            VALUES (
              v_result_is_number, 
              RPAD(p_org_id,8),
              v_det_limit,
              v_min_detect,                             /* MIN QUANT LIMIT */
              v_max_detect,                             /* MAX QUANT LIMIT */
              v_result.rs_detect_comment                /* DESC TEXT */,
              RPAD(USER,8)                              /* D_USERID_CODE */,
              SYSDATE                                   /* D_LAST_UPDATE_TS */,
              v_result.rs_det_limit_uom_nbr             /* TSRUOM_IS_NUMBER */,
              v_result.rs_det_limit_uom_org             /* TSRUOM_ORG_ID */ 
              );

            COMMIT;
          EXCEPTION
            WHEN OTHERS THEN
              sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - writing detect and quantification limit record.');
          END;

        END IF;

      EXCEPTION
        WHEN OTHERS THEN
          sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - getting detect and quantification limit information.');
        
      END;    

      /* keep track of the number of results exported for an activity (incremented only
         when all inserts have been performed so that the value accurately reflects what has
         been loaded into STORET */
      v_result_cnt := v_result_cnt + 1;
      p_act_result_cnt := p_act_result_cnt + 1;
       
    EXCEPTION
      WHEN OTHERS THEN 
        sim2_export_pkg.WRITE_ERROR(p_export_log_seq, SQLERRM||' -  writing result ');
    END;
     
    v_result_rec := v_result_copy_rec;

  END LOOP;

  /* find out if any results were exported on this activity and write error message if none were */
  IF p_act_result_cnt = 0 THEN
    OPEN c_get_activity_id(p_fa_seq);
    FETCH c_get_activity_id INTO v_activity_id;
    CLOSE c_get_activity_id;
  
    sim2_export_pkg.write_error(v_export_log_seq, 'No results exported for activity: '||v_activity_id);
  END IF;
     
END EXPORT_RESULTS;
      
/******************************************************************************************************/
PROCEDURE runtime IS
  
  v_start BINARY_INTEGER;
  v_end BINARY_INTEGER;
  v_temp  NUMBER(10);
  
  CURSOR c_temp IS
SELECT (MAX(TSRFDACT_IS_NUMBER))   
FROM
 TSRFDACT  WHERE TSRFDACT_ORG_ID = RPAD('UTAHDWQ ',8);

BEGIN

  v_start := DBMS_UTILITY.GET_TIME;
  
--  sim_export('UTAHDWQ ',10186);

OPEN c_temp;
FETCH c_temp INTO v_temp;
CLOSE c_temp;
  
  v_end := DBMS_UTILITY.GET_TIME;
  
  --DBMS_OUTPUT.PUT_LINE(v_end - v_start);

END runtime;

/******************************************************************************************************/

END sim2_export_result_pkg;
/
